<?php
namespace Newland\Toubiz\Events\Neos\DataSource;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Neos\Service\DataSource\AbstractDataSource;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Neos\Flow\Annotations as Flow;

/**
 * @Flow\Scope("singleton")
 */
class EventLocationZipDataSource extends AbstractDataSource
{
    static protected $identifier = 'newland-toubiz-events-neos-location-zip';

    /**
     * @var EventRepository
     * @Flow\Inject
     */
    protected $eventRepository;

    /**
     * Get data
     *
     * The return value must be JSON serializable data structure.
     *
     * @param NodeInterface $node The node that is currently edited (optional)
     * @param array $arguments Additional arguments (key / value)
     * @return mixed JSON serializable data
     * @api
     */
    public function getData(NodeInterface $node = null, array $arguments = [])
    {
        $labelFields = [ 'location.zip', 'location.city' ];
        $query = $this->eventRepository->createQueryBuilder('event');

        return $query
            ->join('event.location', 'location')
            ->select([
                $this->getConcatStatement($labelFields) . ' as label',
                'location.zip as value'
            ])
            ->where($query->expr()->andX(
                $query->expr()->notIn('location.zip', [null, '', '0']),
                $query->expr()->notIn('location.city', [null, ''])
            ))
            ->groupBy('location.zip')
            ->getQuery()
            ->execute();
    }

    /**
     * Returns a CONCAT statement that concatenates all of the given fields with spaces
     * separating them. Additionally all fields are coalesced into a string because MySQL
     * returns `null` if any of the components of a `CONCAT` statement are `null`.
     *
     * WARNING: Only use this method with developer provided, fixed strings. Do never (not ever) use
     *          user provided data as input for this method!
     *
     * @example
     * $this->getConcatStatement(['foo', 'bar', 'baz']);
     * // => CONCAT(COALESCE(foo, ''), ' ', COALESCE(bar, ''), ' ', COALESCE(baz, ''))
     *
     * @param string[] $fields
     * @return string
     */
    private function getConcatStatement(array $fields): string
    {
        $concatFields = array_map(function (string $field) {
            return 'COALESCE(' . $field . ', \'\')';
        }, $fields);
        return 'CONCAT(' . implode(', \' \', ', $concatFields) . ')';
    }
}
