<?php declare(strict_types=1);


namespace Newland\Toubiz\Events\Neos\Tests\Integration;


use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Http\Request;
use Neos\Flow\Http\Response;
use Neos\Flow\Http\Uri;
use Neos\Flow\Mvc\ActionRequest;
use Neos\Flow\Mvc\Exception\NoMatchingRouteException;
use Neos\Flow\Mvc\Exception\StopActionException;
use Neos\Neos\Controller\Service\NodesController;
use Neos\Neos\Domain\Service\ContentContextFactory;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Model\Category;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Tests\Factory\CategoryFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\ArticleFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class TranslatesReferencesInNodesWhenTranslatingNodeTest extends IntegrationTestCase
{
    protected $testableSecurityEnabled = true;

    /** @var ContentContextFactory */
    protected $contextFactory;

    /** @var Event */
    protected $de;

    /** @var Event */
    protected $fr;

    /** @var Article */
    protected $cityDe;

    /** @var Article */
    protected $cityFr;

    /** @var Category */
    protected $categoryDe;

    /** @var Category */
    protected $categoryFr;

    public function setUp(): void
    {
        parent::setUp();

        // In this test we don't care about permissions
        $this->privilegeManager->setOverrideDecision(true);
        $this->contextFactory = $this->objectManager->get(ContentContextFactory::class);

        $categoryFactory = new CategoryFactory($this->objectManager);
        $this->categoryDe = $categoryFactory->create([ 'language' => 'de', 'languageGrouping' => 'category-foo' ]);
        $this->categoryFr = $categoryFactory->create([ 'language' => 'fr', 'languageGrouping' => 'category-foo' ]);

        $articleFactory = new ArticleFactory($this->objectManager);
        $this->cityDe = $articleFactory->create([ 'language' => 'de', 'languageGrouping' => 'foobar-city', 'client' => 'foo-bar' ]);
        $this->cityFr = $articleFactory->create([ 'language' => 'fr', 'languageGrouping' => 'foobar-city', 'client' => 'foo-bar' ]);

        $this->de = $this->eventFactory->create([
            'language' => 'de',
            'client' => 'foo-bar',
            'languageGrouping' => 'foo-bar',
            'categories' => [ $this->categoryDe ],
        ]);
        $this->fr = $this->eventFactory->create([
            'language' => 'fr',
            'client' => 'foo-bar',
            'languageGrouping' => 'foo-bar',
            'categories' => [ $this->categoryFr ],
        ]);
    }

    public function testTranslatesTeaserReference(): void
    {
        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:Teaser'));
        $this->node->setProperty('event', $this->de->getPersistenceObjectIdentifier());
        $this->persistNode($this->node);

        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('event', $translatedProperties);
        $this->assertEquals($this->fr->getPersistenceObjectIdentifier(), $translatedProperties['event']);
    }

    public function testUnsetsPropertyIfNoDetailTranslationFound(): void
    {
        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:Teaser'));
        $this->node->setProperty('event', $this->de->getPersistenceObjectIdentifier());
        $this->persistNode($this->node);

        // Italian event does not exist
        $translatedNode = $this->sendTranslationRequestToNeosBackendService('it');
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayNotHasKey('event', $translatedProperties);
    }

    public function testTranslatesPreselectedCitiesInLists(): void
    {
        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:List'));
        $this->node->setProperty('preselectedCities', [ $this->cityDe->getPersistenceObjectIdentifier() ]);
        $this->persistNode($this->node);

        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('preselectedCities', $translatedProperties);
        $this->assertEquals($this->cityFr->getPersistenceObjectIdentifier(), $translatedProperties['preselectedCities'][0]);
    }


    public function testTranslatesCategoryReferencesInList(): void
    {
        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:List'));
        $this->node->setProperty('preselectedCategories', [ $this->categoryDe->getPersistenceObjectIdentifier() ]);
        $this->persistNode($this->node);
        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('preselectedCategories', $translatedProperties);
        $this->assertEquals($this->categoryFr->getPersistenceObjectIdentifier(), $translatedProperties['preselectedCategories'][0]);
    }


    public function testRetainsEventReferencesWithoutLanguage(): void
    {
        $event = $this->eventFactory->create([ 'language' => null ]);

        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:Teaser'));
        $this->node->setProperty('event', $event->getPersistenceObjectIdentifier());
        $this->persistNode($this->node);

        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('event', $translatedProperties);
        $this->assertEquals($event->getPersistenceObjectIdentifier(), $translatedProperties['event']);
    }

    public function testRetainsCitiesReferencesWithoutLanguage(): void
    {
        $untranslatedArticle = (new EventFactory($this->objectManager))->create([ 'language' => null ]);

        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:FilteredLists'));
        $this->node->setProperty('preselectedCities', [ $untranslatedArticle->getPersistenceObjectIdentifier() ]);
        $this->persistNode($this->node);

        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('preselectedCities', $translatedProperties);
        $this->assertEquals($untranslatedArticle->getPersistenceObjectIdentifier(), $translatedProperties['preselectedCities'][0]);
    }

    public function testRetainsCategoryReferencesWithoutLanguage(): void
    {
        $untranslatedCategory = (new EventFactory($this->objectManager))->create([ 'language' => null ]);

        $this->node->setNodeType($this->nodeTypeManager->getNodeType('Newland.Toubiz.Events.Neos:FilteredLists'));
        $this->node->setProperty('preselectedCategories', [ $untranslatedCategory->getPersistenceObjectIdentifier() ]);
        $this->persistNode($this->node);

        $translatedNode = $this->sendTranslationRequestToNeosBackendService();
        $translatedProperties = $translatedNode->getProperties();

        $this->assertArrayHasKey('preselectedCategories', $translatedProperties);
        $this->assertEquals($untranslatedCategory->getPersistenceObjectIdentifier(), $translatedProperties['preselectedCategories'][0]);
    }


    private function sendTranslationRequestToNeosBackendService(string $target = 'fr'): ?NodeInterface
    {
        // Note: Not sending request through $this->browser since that seems to modify the global state causing tests
        //       after this test case to fail.
        $request = new ActionRequest(Request::create(new Uri('http://localhost/neos/service/nodes'), 'POST'));
        $request->setControllerActionName('create');
        $request->setArguments([
            'identifier' => $this->documentNode->getIdentifier(),
            'dimensions' => [ 'language' => [ $target ] ],
            'sourceDimensions' => [ 'language' => [ 'de' ] ],
            'mode' => 'adoptFromAnotherDimensionAndCopyContent',
        ]);

        /** @var NodesController $controller */
        $controller = $this->objectManager->get(NodesController::class);
        try {
            $controller->processRequest($request, new Response());
        } catch (StopActionException $e) {
            // Do nothing
        } catch (NoMatchingRouteException $e) {
            // Do nothing
        }

        return $this->getTranslatedNode($target);
    }

    private function getTranslatedNode(string $language, string $nodeId = null): ?NodeInterface
    {
        $nodeId = $nodeId ?? $this->node->getIdentifier();

        $contextProperties = [
            'workspaceName' => 'live',
            'invisibleContentShown' => true,
            'inaccessibleContentShown' => true,
            'dimensions' => [ 'language' => [ $language ] ],
            'targetDimensions' => [ 'language' => $language ],
        ];

        return $this->contextFactory->create($contextProperties)
            ->getNodeByIdentifier($nodeId);
    }
}
