<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\I18n\Locale;
use Neos\Flow\I18n\Service;

use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\CategoryRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventDateRepository;
use Newland\Toubiz\Sync\Neos\Domain\Filter\CategoryFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventFilter;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventDateFilter;
use Newland\Toubiz\Events\Neos\Utility\DateTimeUtility;

/**
 * Events controller.
 *
 * Responsible for handling event data.
 *
 * @Flow\Scope("singleton")
 */
class EventsController extends AbstractActionController
{
    /**
     * @var int Items per page for pagination.
     */
    const ITEMS_PER_PAGE = 10;

    /**
     * @Flow\Inject
     * @var CategoryRepository
     */
    protected $categoryRepository;

    /**
     * @Flow\Inject
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * @Flow\Inject
     * @var EventDateRepository
     */
    protected $eventDateRepository;

    /**
     * @Flow\Inject
     * @var Service
     */
    protected $i18nService;

    /**
     * Teaser action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserAction()
    {
        $today = DateTimeUtility::buildDateTimeObject();

        $eventFilter = new EventDateFilter;
        $eventFilter->setFromDate($today)
                    ->setOrderBy(['eventDate.beginsAt' => 'asc'])
                    ->setLimit($this->properties['recordLimit']);

        if ($this->properties['filterHighlights']) {
            $eventFilter->setHighlight(true);
        }

        $this->view->assignMultiple([
            'currentDate' => $today,
            'eventDates' => $this->eventDateRepository->findByFilter($eventFilter)
        ]);

        if (array_key_exists('targetNode', $this->properties)) {
            $targetNode = $this->linkingService->convertUriToObject(
                $this->properties['targetNode'],
                $this->node
            );
            $this->view->assign('targetNode', $targetNode);
        }
    }

    /**
     * Index action.
     *
     * Displays an overview of events based on integration type.
     *
     * @return void
     */
    public function indexAction()
    {
        // Assign filter facets for jumping to the filterAction().
        $this->assignFilterFacets();
        $this->view->assign('pageSize', static::ITEMS_PER_PAGE);

        /*
         * Prepare highlight events for all integration types.
         */
        $today = DateTimeUtility::buildDateTimeObject();
        $highlightFilter = new EventDateFilter;
        $highlightFilter->setHighlight(true)
                        ->setFromDate($today)
                        ->setOrderBy(['event.beginsAt' => 'asc'])
                        ->setGroupBy(['event.Persistence_Object_Identifier'])
                        ->setLimit(9);

        $this->view->assign(
            'highlightEventDates',
            $this->eventDateRepository->findByFilter($highlightFilter)
        );

        /*
         * Prepare integration type specific data.
         */
        $integrationTypeAction = 'indexActionFor' . ucwords($this->integrationType) . 'IntegrationType';
        $this->$integrationTypeAction();
    }

    /**
     * Search result action.
     *
     * Displays events based on selected filters.
     *
     * @param array $search
     * @return void
     */
    public function searchResultAction(array $search = [])
    {
        $this->assignFilterFacets($search);
        $page = array_key_exists('page', $search) ? (int)$search['page'] : 1;

        $filter = new EventDateFilter;
        $filter->initialize($search)
               ->setOffset(($page - 1) * self::ITEMS_PER_PAGE)
               ->setLimit(self::ITEMS_PER_PAGE)
               ->setOrderBy(['eventDate.beginsAt' => 'asc']);

        $eventsCount = $this->eventDateRepository->countByFilter($filter, self::ITEMS_PER_PAGE);

        $this->view->assignMultiple([
            'eventDates' => $this->eventDateRepository->findByFilter($filter),
            'pagination' => [
                'isFirst' => ($page == 1),
                'page' => $page,
                'isLast' => $eventsCount['pages'] <= $page,
                'count' => $eventsCount
            ],
            'pageSize' => static::ITEMS_PER_PAGE,
            'search' => $search,
        ]);
    }

    /**
     * Show action.
     *
     * Displays the given event.
     *
     * @param Event $event
     * @return void
     */
    public function showAction(Event $event = null)
    {
        if (!$event) {
            return;
        }

        $this->view->assign('event', $event);
    }

    /**
     * Get events by given filter string.
     *
     * Due to the nature of the request, the current node must be
     * passed manually to the view.
     *
     * @param Node $node The document node for linking to it.
     * @param string $filter JSON string with filtering options.
     * @return string XML
     */
    public function getByFilterAction(Node $node, string $filter = '')
    {
        $dateFilter = new EventDateFilter;

        // Ensure a date is given and set defaults.
        $dateFilter->setFromDate(DateTimeUtility::buildDateTimeObject())
                   ->setOrderBy(['eventDate.beginsAt' => 'asc']);

        // Initialize filter with given search parameters to override defaults.
        $dateFilter->initialize(json_decode($filter, true));

        $eventDates = $this->eventDateRepository->findByFilter($dateFilter);
        $count = $this->eventDateRepository->countByFilter($dateFilter, $dateFilter->getLimit());
        $this->view->assignMultiple([
            'eventDates' => $eventDates,
            'node' => $node,
            'documentNode' => $node,
            'count' => $count,
        ]);

        $dimensions = $node->getContext()->getDimensions();
        if (array_key_exists('language', $dimensions) && $dimensions['language'] !== array()) {
            $currentLocale = new Locale($dimensions['language'][0]);
            $this->i18nService->getConfiguration()->setCurrentLocale($currentLocale);
            $this->i18nService->getConfiguration()->setFallbackRule([
                'strict' => false,
                'order' => array_reverse($dimensions['language'])
            ]);
        }
    }

    /**
     * Prepares index action for 'point' integration type.
     *
     * @return void
     */
    protected function indexActionForPointIntegrationType()
    {
        /*
         * Prepare events for "today".
         */
        $today = DateTimeUtility::buildDateTimeObject();
        $todayFilter = new EventDateFilter;
        $todayFilter->setFromDate($today)
                    ->setToDate($today)
                    ->setOrderBy(['eventDate.beginsAt' => 'asc']);

        $this->view->assign(
            'eventsToday',
            $this->eventDateRepository->findByFilter($todayFilter)
        );

        /*
         * Prepare events for "the next 7 days".
         */
        $nextStart = clone $today;
        $nextStart->modify('+1 day');
        $nextEnd = clone $nextStart;
        $nextEnd->modify('+7 days');

        $nextFilter = new EventDateFilter;
        $nextFilter->setFromDate($nextStart)
                   ->setFromMaxDate($nextEnd)
                   ->setOrderBy(['eventDate.beginsAt' => 'asc']);

        $this->view->assign(
            'nextEventDates',
            $this->eventDateRepository->findByFilter($nextFilter)
        );
    }

    /**
     * Prepares index action for 'destination' integration type.
     *
     * @return void
     */
    protected function indexActionForDestinationIntegrationType()
    {
        /*
         * Prepare the next few events.
         *
         * ATTENTION: The 'limit' parameter should be the same
         * as the one used for pagination in searchResultAction().
         */
        $today = DateTimeUtility::buildDateTimeObject();
        $nextFilter = new EventDateFilter;
        $nextFilter->setFromDate($today)
                   ->setOrderBy(['eventDate.beginsAt' => 'asc'])
                   ->setLimit(self::ITEMS_PER_PAGE);

        $this->view->assign(
            'nextEventDates',
            $this->eventDateRepository->findByFilter($nextFilter)
        );
    }

    /**
     * Fetches and assigns facets required for filtering.
     *
     * @return void
     */
    protected function assignFilterFacets(array $search = [])
    {
        // Inner-joining events and dates to filter categories not having event dates.
        $filter = new CategoryFilter;
        $filter->addInnerJoin('category.events', 'event')
               ->addInnerJoin('event.eventDates', 'eventDate')
               ->setOrderBy(['category.title' => 'ASC']);

        $today = (new \DateTime)->format('Y-m-d');
        $tomorrow = (new \DateTime)->modify('+1 day')->format('Y-m-d');

        $datePickerMinDate = (new \DateTime)->format('m/d/Y');
        $datePickerIsActive = array_key_exists('fromDate', $search)
                              && array_key_exists('toDate', $search)
                              && $search['fromDate'] !== $search['toDate'];

        $this->view->assignMultiple([
            'categories' => $this->categoryRepository->findByFilter($filter),
            'displayDatePicker' => $this->settings['integrationType'] !== 'point',
            'today' => $today,
            'tomorrow' => $tomorrow,
            'datePickerMinDate' => $datePickerMinDate,
            'datePickerIsActive' => $datePickerIsActive,
            'initialDatepickerState' => json_encode([
                'start' => $datePickerIsActive ? @$search['fromDate'] : null,
                'end' => $datePickerIsActive ? @$search['toDate'] : null,
            ])
        ]);
    }
}
