<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Neos\Tests\Unit\Search\Indexer;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ObjectManager;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Newland\Contracts\Neos\Search\SearchBackend;
use Newland\Contracts\Neos\Search\SearchRequest;
use Newland\Contracts\Neos\Search\SearchResultCollection;
use Newland\Toubiz\Events\Neos\Search\Indexer\EventIndexer;
use Newland\Toubiz\Search\Neos\Progress\NullHandler;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventDateFactory;
use Newland\Toubiz\Sync\Neos\Tests\Factory\EventFactory;

class EventIndexerTest extends FunctionalTestCase implements SearchBackend
{
    protected static $testablePersistenceEnabled = true;

    /** @var EventIndexer */
    protected $subject;

    /** @var EventFactory */
    protected $eventFactory;

    /** @var EventDateFactory */
    protected $eventDateFactory;

    /** @var IndexRecordModification[] */
    protected $entries = [];

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = $this->objectManager->get(EventIndexer::class);
        $this->eventFactory = new EventFactory($this->objectManager);
        $this->eventDateFactory = new EventDateFactory($this->objectManager);
        $this->entries = [];
    }

    public function testIndexesEvent(): void
    {
        $event = $this->createEventAt(
            (new \DateTime())->add(new \DateInterval('P1D')),
            (new \DateTime())->add(new \DateInterval('P1DT1H')),
            [ 'title' => '__SOME_TITLE__', 'description' => '__SOME_DESCRIPTION__' ]
        );

        $this->subject->index($this, new NullHandler());

        $this->assertCount(1, $this->entries);
        $this->assertEquals($event->getPersistenceObjectIdentifier(), $this->entries[0]->getIdentifier());
        $this->assertStringContainsString('__SOME_TITLE__', $this->entries[0]->getContent());
        $this->assertStringContainsString('__SOME_DESCRIPTION__', $this->entries[0]->getContent());
    }

    public function testDoesNotIndexPastEvents(): void
    {
        $this->createEventAt(
            (new \DateTime())->sub(new \DateInterval('P1D')),
            (new \DateTime())->sub(new \DateInterval('P1DT1H')),
            [ 'title' => '__SOME_TITLE__', 'description' => '__SOME_DESCRIPTION__' ]
        );

        $this->subject->index($this, new NullHandler());

        $this->assertCount(0, $this->entries);
    }

    public function testIndexesAdditionalSearchStringForEvent(): void
    {
        $this->createEventAt(
            (new \DateTime())->add(new \DateInterval('P1D')),
            (new \DateTime())->add(new \DateInterval('P1DT1H')),
            [ 'additionalSearchString' => '__ADDITIONAL_SEARCH_STRING__' ]
        );

        $this->subject->index($this, new NullHandler());

        $this->assertCount(1, $this->entries);
        $this->assertStringContainsString('__ADDITIONAL_SEARCH_STRING__', $this->entries[0]->getContent());
    }

    public function initialize(): void
    {
        // Mock implementation of search backend
    }

    public function setSource(string $source): void
    {
        // Mock implementation of search backend
    }

    public function createOrUpdateIndexEntry(IndexRecordModification $modification): void
    {
        $this->entries[] = $modification;
    }

    public function afterIndexing(): void
    {
        // Mock implementation of search backend
    }

    public function deleteObsoleteIndexEntries(array $identifiersToRetain): void
    {
        // Mock implementation of search backend
    }

    public function search(SearchRequest $request): SearchResultCollection
    {
        // Mock implementation of search backend
    }

    private function createEventAt(\DateTime $beginsAt, \DateTime $endsAt, array $properties = [], array $dateProperties = []): Event
    {
        $properties['beginsAt'] = $beginsAt;
        $properties['endsAt'] = $beginsAt;
        $event = $this->eventFactory->create($properties);

        $dateProperties['beginsAt'] = $beginsAt;
        $dateProperties['endsAt'] = $beginsAt;
        $dateProperties['event'] = $event;
        $this->eventDateFactory->create($dateProperties);

        $this->persistenceManager->persistAll();
        return $this->persistenceManager->getObjectByIdentifier($event->getPersistenceObjectIdentifier(), Event::class);
    }
}
