<?php declare(strict_types=1);

namespace Newland\Toubiz\Events\Neos\Tests\Integration;

use Doctrine\ORM\EntityManagerInterface;
use Neos\Flow\Utility\Now;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;

class DetailPageTest extends IntegrationTestCase
{

    /** @var Event */
    protected $event;

    /** @var EventDate[] */
    protected $dates;


    public function setUp(): void
    {
        parent::setUp();

        $this->objectManager->setInstance(Now::class, new Now('2019-01-01T06:00'));

        $this->dates = [
            $this->eventDateFactory->create([ 'beginsAt' => '2019-01-01T14:00', 'endsAt' => '2019-01-01T17:00' ]),
            $this->eventDateFactory->create([ 'beginsAt' => '2019-01-02T12:00', 'endsAt' => '2019-01-02T15:00' ]),
            $this->eventDateFactory->create([ 'beginsAt' => '2019-01-03T06:00', 'endsAt' => '2019-01-03T18:00' ]),
        ];

        $this->event = $this->eventFactory->create([
            'title' => 'foo bar test',
            'eventDates' => $this->dates,
        ]);
    }

    public function tearDown(): void
    {
        $this->objectManager->forgetInstance(Now::class);
        parent::tearDown();
    }

    public function testRendersDetailView(): void
    {
        $response = $this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
        );
        $this->assertResponseOk($response);
    }

    public function testRendersDetailViewForDates(): void
    {
        $this->assertResponseOk($this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode(
                $this->event,
                $this->documentNode,
                $this->dates[0]
            )
        ));
        $this->assertResponseOk($this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode(
                $this->event,
                $this->documentNode,
                $this->dates[1]
            )
        ));
        $this->assertResponseOk($this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode(
                $this->event,
                $this->documentNode,
                $this->dates[2]
            )
        ));
    }

    // See TBEVENTS-132
    public function testRendersTimesOfRecurringEventsCorrectly(): void
    {
        $response = $this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
        );
        $this->assertResponseOk($response);

        $bodyWithoutTags = preg_replace(
            '/\s+/',
            ' ',
            strip_tags((string) $response->getBody())
        );
        $this->assertContains('01.01.2019 | 14:00 – 17:00', $bodyWithoutTags);
        $this->assertContains('02.01.2019 | 12:00 – 15:00', $bodyWithoutTags);
        $this->assertContains('03.01.2019 | 06:00 – 18:00', $bodyWithoutTags);
    }

    public function testRendersNoTimeIfThereIsNoSpecificTime(): void
    {
        $this->dates[0]->setBeginsAt(new \DateTime('2055-01-01T00:00'));
        $this->dates[0]->setBeginsAtSpecificTime(false);
        $this->dates[0]->setEndsAt(new \DateTime('2055-01-01T23:59'));
        $this->dates[0]->setEndsAtSpecificTime(false);
        $this->persist($this->dates[0]);

        $response = $this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
        );
        $this->assertResponseOk($response);

        $body = preg_replace('/\s+/', ' ', (string) $response->getBody());
        $this->assertStringNotContainsString('00:00', $body);
        $this->assertStringNotContainsString('23:59', $body);
    }

    public function testRendersNoEndTimeIfItIsNotSpecific(): void
    {
        $this->dates[0]->setBeginsAt(new \DateTime('2055-01-01T14:00'));
        $this->dates[0]->setBeginsAtSpecificTime(true);
        $this->dates[0]->setEndsAt(new \DateTime('2055-01-01T23:59'));
        $this->dates[0]->setEndsAtSpecificTime(false);
        $this->persist($this->dates[0]);

        $response = $this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
        );
        $this->assertResponseOk($response);

        $body = preg_replace('/\s+/', ' ', (string) $response->getBody());
        $this->assertStringContainsString('14:00', $body);
        $this->assertStringNotContainsString('23:59', $body);
    }

    public function testDoesNotRenderAnyTimeIfStartTimeIsNotSpecific(): void
    {
        $this->dates[0]->setBeginsAt(new \DateTime('2055-01-01T00:00'));
        $this->dates[0]->setBeginsAtSpecificTime(false);
        $this->dates[0]->setEndsAt(new \DateTime('2055-01-01T14:42'));
        $this->dates[0]->setEndsAtSpecificTime(true);
        $this->persist($this->dates[0]);

        $response = $this->browser->request(
            $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
        );
        $this->assertResponseOk($response);

        $body = preg_replace('/\s+/', ' ', (string) $response->getBody());
        $this->assertStringNotContainsString('00:00', $body);
        $this->assertStringNotContainsString('14:42', $body);
    }

    private function persist($entity): void
    {
        /** @var EntityManagerInterface $em */
        $em = $this->objectManager->get(EntityManagerInterface::class);
        $em->persist($entity);
        $em->flush($entity);
    }

    public function testDisplaysSourceIfNotDisabled(): void
    {
        $sourceName = sprintf('__SOURCE__NAME__%s__', md5(random_bytes(32)));
        $this->event->setSourceName($sourceName);

        $this->withMockedConfiguration(
            [ 'Newland.Toubiz.Events.Neos.displayOptions.show.hide' => [ ] ],
            function() use ($sourceName) {
                $response = $this->browser->request(
                    $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
                );
                $this->assertResponseOk($response);
                $this->assertResponseContains($sourceName, $response);
            }
        );
    }

    public function testDoesNotDisplaySourceIfDisabled(): void
    {
        $sourceName = sprintf('__SOURCE__NAME__%s__', md5(random_bytes(32)));
        $this->event->setSourceName($sourceName);

        $this->withMockedConfiguration(
            [ 'Newland.Toubiz.Events.Neos.displayOptions.show.hide' => [ 'source' ] ],
            function() use ($sourceName) {
                $response = $this->browser->request(
                    $this->eventUrlService->generateUrlByCurrentNode($this->event, $this->documentNode)
                );
                $this->assertResponseOk($response);
                $this->assertResponseNotContains($sourceName, $response);
            }
        );
    }

}
