<?php declare(strict_types=1);
namespace Newland\Toubiz\Events\Neos\Search\Indexer;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Contracts\Neos\Search\IndexerInterface;
use Newland\Contracts\Neos\Search\SearchBackend;
use Newland\Contracts\Neos\Search\IndexRecordModification;
use Newland\Contracts\Neos\Search\ProgressHandler;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventFilter;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;

/**
 * Event indexer.
 *
 * Indexes events for the search.
 *
 * @Flow\Scope("singleton")
 */
class EventIndexer implements IndexerInterface
{
    public const SCOPE = 'event';

    /**
     * @var EventRepository
     * @Flow\Inject()
     */
    protected $eventRepository;

    /** @var int|null */
    private $total = null;

    /** @var string[] */
    protected $indexed = [];

    /** @var array */
    protected $configuration;

    public function index(SearchBackend $backend, ProgressHandler $progressHandler): void
    {
        $this->eventRepository->withoutLanguageHandling(function () use ($progressHandler, $backend) {
            $filter = $this->filter();
            $this->total = $this->eventRepository->countByFilter($filter, 1)['items'] ?? null;

            $this->eventRepository->forEachFilter($filter, function (Event $event) use ($progressHandler, $backend) {
                $modification = $this->eventToModification($event);
                $backend->createOrUpdateIndexEntry($modification);
                $this->indexed[] = $modification->getIdentifier();
                $progressHandler->informProgress('Events', \count($this->indexed), $this->total);
            });
        });
    }

    public function postIndex(SearchBackend $backend): void
    {
        $backend->deleteObsoleteIndexEntries($this->indexed);
    }

    public function setConfiguration(array $configuration = []): void
    {
        $this->configuration = $configuration;
    }

    private function eventToModification(Event $event): IndexRecordModification
    {
        $content = [
            $event->getTitle(),
            $event->getDescription(),
        ];

        return new IndexRecordModification(
            (string) $event->getPersistenceObjectIdentifier(),
            $event->getTitle(),
            (string) $event->getDescription(),
            trim(implode(' ', $content)),
            static::SCOPE,
            $event->getLanguage()
        );
    }

    private function filter(): EventFilter
    {
        $today = new \DateTime();
        $today->setTime(0, 0, 0);

        $filter = new EventFilter();
        $filter->setFromDate($today);
        return $filter;
    }
}
