<?php declare(strict_types=1);
namespace Newland\Toubiz\Events\Neos\Service;

use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\ConfigurationManager;
use Newland\NeosCommon\Domain\Model\Topic;
use Newland\NeosCommon\Service\AbstractRegionService;
use Newland\NeosCommon\Service\ConfigurationService;
use Newland\Toubiz\Events\Neos\Domain\Repository\TopicRepository;

class RegionService extends AbstractRegionService
{
    /**
     * @var ConfigurationService
     * @Flow\Inject()
     */
    protected $configurationService;

    protected function getRegionsAvailable(): array
    {
        return (array) $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Events.Neos.regions.available'
        );
    }

    protected function getRegionsEnabled(NodeInterface $node): array
    {
        $configuration = (array) $this->configurationManager->getConfiguration(
            ConfigurationManager::CONFIGURATION_TYPE_SETTINGS,
            'Newland.Toubiz.Events.Neos.regions.enabled'
        );

        if ($configuration['default'] ?? null) {
            $configuration = $this->configurationService->getCurrentSiteConfiguration(
                $node,
                'Newland.Toubiz.Events.Neos.regions.available'
            );
        }

        return $configuration;
    }

    /** @var Topic[] */
    private $regionTopics;

    /**
     * Returns all available regions which are also enabled.
     *
     * @param NodeInterface $node
     * @return Topic[]
     */
    public function getAllEnabledAsTopics(NodeInterface $node): array
    {
        if ($this->regionTopics === null) {
            $regionsEnabled = $this->getRegionsEnabledCached($node);

            $attainable = array_filter(
                $this->getRegionsAvailable(),
                function ($region) use ($regionsEnabled) {
                    return in_array($region, $regionsEnabled);
                },
                ARRAY_FILTER_USE_KEY
            );

            $this->regionTopics = [];
            foreach ($attainable as $region => $config) {
                $this->regionTopics[] = new Topic(
                    TopicRepository::TYPE_REGION,
                    $region,
                    $config['name']
                );
            }
        }

        return $this->regionTopics;
    }

    /**
     * @param NodeInterface $node
     * @param string $identifier
     * @return Topic|null
     */
    public function findByIdentifier(NodeInterface $node, string $identifier)
    {
        foreach ($this->getAllEnabledAsTopics($node) as $region) {
            if ($region->getIdentifier() === $identifier) {
                return $region;
            }
        }
        return null;
    }
}
