<?php
namespace Newland\Toubiz\Events\Neos\Service;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Mvc\Controller\ControllerContext;
use Neos\Neos\Domain\Service\ContentContext;
use Newland\NeosCommon\Service\ControllerContextFactory;
use Newland\PageFrameProvider\Service\PageFrameLinkingService;
use Newland\Toubiz\Events\Neos\Routing\EventDetailRoutePart;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;

/**
 * Generates detail URLs for events.
 *
 * This Service is tightly coupled to the `PageFrameController` functionality of `Newland.PageFrameProvider` to have
 * detail pages that are independent of the PageTree.
 *
 * @Flow\Scope("singleton")
 */
class EventUrlService
{
    /**
     * @var ControllerContextFactory
     * @Flow\Inject()
     */
    protected $controllerContextFactory;

    /**
     * @var PageFrameLinkingService
     * @Flow\Inject()
     */
    protected $pageFrameLinkingService;

    /**
     * @var array
     * @Flow\InjectConfiguration("detailPages")
     */
    protected $configuration;

    public function generateUrl(
        Event $event,
        ControllerContext $context,
        ?NodeInterface $referenceNode = null,
        ?EventDate $forDate = null
    ): ?string {
        return $this->getDateUrl($event, $forDate) ?? $this->generateDetailViewUrl($event, $context, $referenceNode);
    }

    /**
     * Some requests are within a controller context that has no "current node".
     * So we pass the current node manually and create a fake controller context around that.
     *
     * @param Event $event
     * @param Node $node
     * @return mixed|string|null
     */
    public function generateUrlByCurrentNode(Event $event, Node $node, EventDate $forDate = null)
    {
        $context = $this->controllerContextFactory->initializeFakeControllerContext($node);
        $configuration = $this->configuration ?? null;
        if ($configuration) {
            return $this->overrideUrlDetail($node, $event, $context);
        }
        return $this->generateUrl($event, $context, $node, $forDate);
    }

    private function getDateUrl(Event $event, ?EventDate $date = null): ?string
    {
        $date = $date ?? $event->getEventDates()[0] ?? null;
        if ($date) {
            return $date->getDetailUri();
        }
        return null;
    }

    private function generateDetailViewUrl(
        Event $event,
        ControllerContext $context,
        NodeInterface $referenceNode = null
    ): string {
        return $this->pageFrameLinkingService->build(
            $context->getUriBuilder(),
            EventDetailRoutePart::PACKAGE,
            EventDetailRoutePart::CONTROLLER,
            EventDetailRoutePart::ACTION,
            'eventDetail',
            [
                'event' => $event,
                // events don't currently have types
                EventDetailRoutePart::TYPE_ARGUMENT => 'default',
            ],
            EventDetailRoutePart::ARGUMENT_NAMESPACE,
            $referenceNode
        );
    }

    private function overrideUrlDetail(?Node $node, Event $event, ControllerContext $controllerContext): ?string
    {
        if ($node === null) {
            return '';
        }
        /** @var ContentContext $context */
        $context = $node->getContext();
        $site = $context->getCurrentSite();
        $clientConfiguration = $this->configuration['clients'][$site->getName()]
            ?? $this->configuration['clients']['Default'] ?? null;
        if ($clientConfiguration['redirectToTportal'] ?? null) {
            return $this->simpleTemplate(
                $clientConfiguration['detailUri'],
                [
                    'name' => $this->sanitizeTitle($event->getTitle()),
                    'id' => $event->getOriginalId(),
                ]
            );
        }
        return $this->generateUrl($event, $controllerContext, $node);
    }

    private function simpleTemplate(
        string $template,
        array $replacements,
        array $enclosure = [ '{', '}' ]
    ): string {
        foreach ($replacements as $before => $after) {
            $template = str_replace(
                $enclosure[0] . $before . $enclosure[1],
                $after,
                $template
            );
        }

        return $template;
    }

    private function sanitizeTitle(string $title)
    {
        $title = \Safe\preg_replace('/[^A-Za-z0-9\-\s]/', '', $title);
        $title = str_replace(' ', '-', $title);
        return $title;
    }
}
