<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\Configuration\Exception\InvalidConfigurationException;
use Newland\Toubiz\Events\Neos\Domain\Repository\TopicRepository;
use Newland\Toubiz\Events\Neos\Service\QueryOverride;
use Newland\Toubiz\Events\Neos\Service\FilterConfiguration;
use Newland\Toubiz\Events\Neos\Service\RegionService;
use Newland\Toubiz\Events\Neos\Utility\IntegerUtility;
use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Model\EventDate;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventDateRepository;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventDateFilter;
use Newland\Toubiz\Events\Neos\Utility\DateTimeUtility;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;

/**
 * Events controller.
 *
 * Responsible for handling event data.
 *
 * @Flow\Scope("singleton")
 */
class EventsController extends AbstractActionController
{
    /**
     * @var int Items per page for pagination.
     */
    const ITEMS_PER_PAGE = 10;

    /**
     * @Flow\Inject
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * @Flow\Inject
     * @var EventDateRepository
     */
    protected $eventDateRepository;

    /**
     * @var RegionService
     * @Flow\Inject()
     */
    protected $regionService;

    /**
     * @Flow\Inject
     * @var TopicRepository
     */
    protected $topicRepository;

    /**
     * @var QueryOverride
     * @Flow\Inject()
     */
    protected $queryOverride;

    public function initializeAction(): void
    {
        parent::initializeAction();
        $this->queryOverride->initialize($this->properties);
    }

    /**
     * @var FilterConfiguration
     * @Flow\Inject()
     */
    protected $filterConfiguration;

    /**
     * Teaser action.
     *
     * Show a event on a teaser.
     *
     * @return void
     */
    public function teaserAction()
    {

        $this->view->assignMultiple(
            [
                'event' => $this->eventRepository->findByIdentifier($this->properties['event'] ?? ''),
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
            ]
        );
    }

    /**
     * Teaser List action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserListAction()
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $regions = $this->properties['regions'] ?? [];

        $eventFilter = new EventDateFilter;
        $eventFilter->setFromDate($today)
            ->setLocationZips(
                $this->regionService->collectZipsFromRegionKeys($regions, true)
            )
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ])
            ->setLimit($this->properties['recordLimit']);

        if ($this->properties['filterHighlights']) {
            $eventFilter->setHighlight(true);
        }
        $this->view->assignMultiple(
            [
                'wrapStart' => $this->request->getInternalArgument('__wrapStart'),
                'wrapEnd' => $this->request->getInternalArgument('__wrapEnd'),
                'currentDate' => $today,
                'eventDates' => $this->eventDateRepository->findByFilter($eventFilter),
            ]
        );
    }

    /**
     * Index action.
     *
     * Displays an overview of events based on integration type.
     *
     * @return void
     */
    public function indexAction()
    {
        $this->assignDatePickerProperties();
        $topics = $this->topicRepository->findByIdentifiers($this->properties['topics'] ?? []);

        $filterSections = $this->filterConfiguration->getFilterSections(
            $this->queryOverride->getOverride(),
            $this->properties
        );
        $this->view->assignMultiple(
            [
                'highlightEventDates' => $this->highlightedEvents(),
                'topics' => $topics,
                'queryOverride' => $this->queryOverride->getOverride(),
                'filterSections' => $filterSections,
                'datePickerLoadPage' => $this->getDatePickerLoadPage(),
                'argumentNamespace' => $this->request->getArgumentNamespace(),
            ]
        );

        switch (strtolower($this->integrationType)) {
            case 'point':
                $this->indexActionForPointIntegrationType();
                break;
            case 'destination':
                $this->indexActionForDestinationIntegrationType();
                break;
            default:
                throw new InvalidConfigurationException(sprintf(
                    'Integrationtype "%s" is not supported. Supported integration types: point, destination',
                    $this->integrationType
                ));
        }
    }

    private function indexActionForPointIntegrationType(): void
    {
        $this->view->assignMultiple(
            [
                'eventsToday' => $this->eventsToday(),
                'nextEventDates' => $this->eventsComingUp(),
            ]
        );
    }

    private function indexActionForDestinationIntegrationType(): void
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $demand = $this->queryOverride->computeDemandFromQuery($this->properties);
        $nextFilter = $this->filterForProperties($demand)->setFromDate($today);
        $this->view->assignMultiple(
            [
                'nextEventDates' => $this->eventDateRepository->findByFilter($nextFilter),
                'hasMoreEvents' => $this->eventDateRepository
                        ->countByFilter($nextFilter, self::ITEMS_PER_PAGE)['pages'] > 1,
            ]
        );
    }

    /**
     * @param array $query
     * @return void
     */
    public function searchResultAction(array $query = [])
    {
        $this->assignDatePickerProperties($query);
        $this->assignPropertiesForFilteredLists($query);
    }

    /**
     * @param Node $node
     * @param array $query
     */
    public function searchResultAjaxAction(Node $node, array $query = [])
    {
        $this->setupCustomNodeMappingForAjaxRequest($node);
        $this->assignDatePickerProperties($query);
        $this->assignPropertiesForFilteredLists($query);
    }

    /**
     * @param Event $event
     * @return void
     */
    public function showAction(Event $event = null)
    {
        if (!$event) {
            return;
        }

        $this->view->assign('event', $event);
    }

    private function assignPropertiesForFilteredLists(array $query)
    {
        $demand = $this->queryOverride->computeDemandFromQuery($query);
        $filter = $this->filterForProperties($demand);

        $page = IntegerUtility::normalizeToPositiveValue($query['page'] ?? 1);
        $filterSections = $this->filterConfiguration->getFilterSections(
            $this->queryOverride->getOverride(),
            $this->properties
        );
        $this->view->assignMultiple(
            [
                'filterSections' => $filterSections,
                'eventDates' => $this->eventDateRepository->findByFilter($filter),
                'pagination' => $this->paginationProperties($filter, $page),
                'queryOverride' => $this->queryOverride->getOverride(),
                'query' => $query,
                'argumentNamespace' => $this->request->getArgumentNamespace(),
            ]
        );
    }

    private function assignDatePickerProperties(array $query = [])
    {
        $today = (new \DateTime())->format('Y-m-d');
        $tomorrow = (new \DateTime())->modify('+1 day')->format('Y-m-d');
        $datePickerMinDate = (new \DateTime())->format('m/d/Y');
        $datePickerIsActive = array_key_exists('fromDate', $query)
            && array_key_exists('toDate', $query);

        $this->view->assignMultiple(
            [
                'today' => $today,
                'tomorrow' => $tomorrow,
                'datePickerMinDate' => $datePickerMinDate,
                'datePickerIsActive' => $datePickerIsActive,
                'initialDatepickerState' => json_encode(
                    [
                        'start' => $datePickerIsActive ? ($query['fromDate'] ?? null) : null,
                        'end' => $datePickerIsActive ? ($query['toDate'] ?? null) : null,
                    ]
                ),
            ]
        );
    }

    private function filterForProperties(array $properties): EventDateFilter
    {
        $page = IntegerUtility::normalizeToPositiveValue($properties['page'] ?? 1);
        if (array_key_exists('categories', $properties)) {
            $properties['categories'] = array_values($properties['categories']);
        }
        if (array_key_exists('eventTags', $properties)) {
            $properties['eventTags'] = array_values($properties['eventTags']);
        }
        /** @var EventDateFilter $filter */
        $filter = (new EventDateFilter())
            ->initialize($properties)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ])
            ->setOffset(($page - 1) * self::ITEMS_PER_PAGE)
            ->setLimit(self::ITEMS_PER_PAGE);

        if (array_key_exists('preselectedRegions', $properties)
            && !empty($properties['preselectedRegions'])
        ) {
            $filter->setLocationZips(
                $this->regionService->collectZipsFromRegionKeys($properties['preselectedRegions'], true)
            );
        } elseif (array_key_exists('regions', $properties)) {
            $filter->setLocationZips($this->regionService->getZipsFromRegionKeys($properties['regions']));
        }


        return $filter;
    }

    private function paginationProperties(EventDateFilter $filter, int $page): array
    {
        $eventsCount = $this->eventDateRepository->countByFilter($filter, self::ITEMS_PER_PAGE);
        return [
            'isFirst' => $page === 1,
            'page' => $page,
            'isLast' => $eventsCount['pages'] <= $page,
            'count' => $eventsCount,
        ];
    }

    /**
     * @return EventDate[]
     */
    private function highlightedEvents(): array
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $demand = $this->queryOverride->computeDemandFromQuery([ 'highlight' => true, 'fromDate' => $today ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'event.beginsAt' => 'asc' ])
            ->setGroupBy([ 'event.Persistence_Object_Identifier' ])
            ->setLimit(9);

        return (array) $this->eventDateRepository->findByFilter($filter);
    }

    /**
     * @return EventDate[]
     */
    private function eventsToday(): array
    {
        $today = DateTimeUtility::buildDateTimeObject();
        $demand = $this->queryOverride->computeDemandFromQuery([ 'fromDate' => $today, 'toDate' => $today ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ]);

        return (array) $this->eventDateRepository->findByFilter($filter);
    }

    /**
     * @return EventDate[]
     */
    private function eventsComingUp(): array
    {
        $start = DateTimeUtility::buildDateTimeObject();
        $start->modify('+1 day');
        $end = DateTimeUtility::buildDateTimeObject();
        $end->modify('+7 days');

        $demand = $this->queryOverride->computeDemandFromQuery([ 'fromDate' => $start, 'fromMaxDate' => $end ]);
        $filter = $this->filterForProperties($demand)
            ->setOrderBy([ 'eventDate.beginsAt' => 'asc' ]);

        return (array) $this->eventDateRepository->findByFilter($filter);
    }

    private function getDatePickerLoadPage(): string
    {
        return $this->getControllerContext()->getUriBuilder()
            ->reset()
            ->uriFor(
                'searchResult',
                [
                    'query' => [
                        'fromDate' => '{{=it.start}}',
                        'toDate' => '{{=it.end}}',
                    ],
                ]
            );
    }
}
