<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\ContentRepository\Domain\Model\Node;
use Neos\Flow\Annotations as Flow;

use Newland\Toubiz\Sync\Neos\Domain\Model\Event;
use Newland\Toubiz\Sync\Neos\Domain\Repository\EventRepository;
use Newland\Toubiz\Sync\Neos\Domain\Filter\EventFilter;
use Newland\Toubiz\Events\Neos\Utility\DateTimeUtility;

/**
 * Events controller.
 *
 * Responsible for handling event data.
 *
 * @Flow\Scope("singleton")
 */
class EventsController extends AbstractActionController
{
    /**
     * @Flow\Inject
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * Teaser action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserAction()
    {
        $today = DateTimeUtility::buildDateTimeObject();

        $eventFilter = new EventFilter;
        $eventFilter->setToDate($today)
                    ->setLimit($this->properties['recordLimit']);

        if ($this->properties['filterHighlights']) {
            $eventFilter->setHighlight(true);
        }

        $this->view->assignMultiple([
            'currentDate' => $today,
            'events' => $this->eventRepository->findByFilter($eventFilter)
        ]);

        if (array_key_exists('targetNode', $this->properties)) {
            $targetNode = $this->linkingService->convertUriToObject(
                $this->properties['targetNode'],
                $this->node
            );
            $this->view->assign('targetNode', $targetNode);
        }
    }

    /**
     * Index action.
     *
     * Displays an overview of events with filtering options.
     *
     * @param string $year
     * @param string $month
     * @param string $day
     * @return void
     */
    public function indexAction($year = null, $month = null, $day = null)
    {
        $currentDate = DateTimeUtility::buildDateTimeObject($year, $month, $day);
        $today = DateTimeUtility::buildDateTimeObject();

        $eventFilter = new EventFilter;
        $eventFilter->setFromDate($currentDate)
                    ->setLimit(4);

        $highlightFilter = new EventFilter;
        $highlightFilter->setHighlight(true)
                        ->setFromDate($currentDate)
                        ->setLimit(3);

        $this->view->assignMultiple([
            'currentDate' => $currentDate,
            'today' => $today,
            'currentDateIsToday' => ($currentDate == $today),
            'events' => $this->eventRepository->findByFilter($eventFilter),
            'highlightEvents' => $this->eventRepository->findByFilter($highlightFilter)
        ]);
    }

    /**
     * Show action.
     *
     * Displays the given event.
     *
     * @param Event $event
     * @return void
     */
    public function showAction(Event $event = null)
    {
        $this->view->assign('event', $event);
    }

    /**
     * Get events by given filter string.
     *
     * Due to the nature of the request, the current node must be
     * passed manually to the view.
     *
     * @param Node $node
     * @param string $filter JSON string with filtering options.
     * @return string XML
     */
    public function getByFilterAction(Node $node, string $filter = '')
    {
        $eventFilter = new EventFilter;
        $eventFilter->initialize(json_decode($filter, true));

        $events = $this->eventRepository->findByFilter($eventFilter);
        $this->view->assignMultiple([
            'events' => $events,
            'currentDate' => DateTimeUtility::buildDateTimeObject(),
            'node' => $node
        ]);
    }
}
