<?php
namespace Newland\Toubiz\Events\Neos\Service;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Neos\Flow\Http\Client\CurlEngine;
use Neos\Flow\Http\Request;
use Neos\Flow\Http\Uri;

/**
 * Toubiz service class.
 *
 * Service class to fetch data from the toubiz API.
 *
 * This class is intended to be refactored into a custom
 * package for handling API requests system-independent.
 *
 * @Flow\Scope("singleton")
 */
class ToubizService
{
    /**
     * @var string
     */
    const BASE_URI = 'https://api.toubiz.de/json/event/';

    /**
     * @var CurlEngine
     */
    protected $client;

    /**
     * @Flow\InjectConfiguration
     * @var array
     */
    protected $configuration;

    /**
     * Class constructor.
     *
     * @return void
     */
    public function initializeObject()
    {
        if (empty($this->configuration['apiKey'])) {
            throw new \UnexpectedValueException('The API key for the toubiz service is missing!');
        }

        if (empty($this->configuration['client'])) {
            throw new \UnexpectedValueException('The client for the toubiz service is missing!');
        }

        $this->client = new CurlEngine;
    }

    /**
     * Gets the full list of upcoming events.
     *
     * This list only contains abstract information.
     *
     * @return array<stdObject>
     */
    public function getUpcomingList()
    {
        $params = ['all_in' => 1];
        $data = $this->requestData('json_result.php', $params);
        return $data->result;
    }

    /**
     * Gets details for a given event.
     *
     * @param int $originalId The ID toubiz uses
     * @return array
     */
    public function getDetails($originalId)
    {
        $params = ['id' => $originalId];
        $data = $this->requestData('json_detail.php', $params);
        return $data;
    }

    /**
     * Requests data based on given target and query params.
     *
     * @param string $target The target script
     * @param array $queryParams Additional query parameters
     * @return Response|boolean
     */
    protected function requestData($target, $queryParams)
    {
        $queryParams['mandant'] = $this->configuration['client'];
        $queryParams['key'] = $this->configuration['apiKey'];

        if (!empty($this->configuration['organizerRestriction'])) {
            $queryParams['organizer'] = $this->configuration['organizerRestriction'];
        }

        $rawUri = self::BASE_URI . $target . '?' . http_build_query($queryParams);
        $uri = new Uri($rawUri);
        $request = Request::create($uri, 'GET');
        $response = $this->client->sendRequest($request);
        if ($response->getStatusCode() == 200) {
            $data = json_decode($response->getContent());
            return $data;
        } else {
            return false;
        }
    }
}
