<?php
namespace Newland\Toubiz\Events\Neos\Controller;

/*
 * This file is part of the "toubiz-events-neos" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;

use Newland\Toubiz\Events\Neos\Domain\Model\Event;
use Newland\Toubiz\Events\Neos\Domain\Repository\EventRepository;

/**
 * Events controller.
 *
 * Responsible for handling event data.
 *
 * @Flow\Scope("singleton")
 */
class EventsController extends AbstractActionController
{
    /**
     * @Flow\Inject
     * @var EventRepository
     */
    protected $eventRepository;

    /**
     * Teaser action.
     *
     * Lists a subset of upcoming events.
     *
     * @return void
     */
    public function teaserAction()
    {
        $limit = $this->properties['recordLimit'];
        $highlights = $this->properties['filterHighlights'];
        $this->view->assign('events', $this->eventRepository->findUpcoming($limit, $highlights));
        $this->view->assign('currentDate', $this->getCurrentDate());

        if (array_key_exists('targetNode', $this->properties)) {
            $targetNode = $this->linkingService->convertUriToObject(
                $this->properties['targetNode'],
                $this->node
            );
            $this->view->assign('targetNode', $targetNode);
        }
    }

    /**
     * Index action.
     *
     * Displays an overview of events with filtering options.
     *
     * @param string $year
     * @param string $month
     * @param string $day
     * @return void
     */
    public function indexAction($year = null, $month = null, $day = null)
    {
        $currentDate = $this->getCurrentDate($year, $month, $day);
        $today = new \DateTime;
        $today->setTime(0, 0);

        $this->view->assignMultiple([
            'currentDate' => $currentDate,
            'today' => $today,
            'currentDateIsToday' => ($currentDate == $today),
            'events' => $this->eventRepository->findByDate($currentDate)
        ]);
    }

    /**
     * Show action.
     *
     * Displays the given event.
     *
     * @param Event $event
     * @return void
     */
    public function showAction(Event $event = null)
    {
        $this->view->assign('event', $event);
    }

    /**
     * Builds the current date based on given params, or "today".
     *
     * @param string $year
     * @param string $month
     * @param string $day
     * @return \DateTime
     */
    protected function getCurrentDate($year = null, $month = null, $day = null)
    {
        $currentDate = new \DateTime;
        $currentDate->setDate(
            ($year ?? date('Y')),
            ($month ?? date('m')),
            ($day ?? date('d'))
        );
        $currentDate->setTime(0, 0);
        return $currentDate;
    }
}
