<?php declare(strict_types=1);

namespace Newland\Toubiz\Common\Neos\Translation;

use Neos\ContentRepository\Domain\Model\Node;
use Neos\ContentRepository\Domain\Model\NodeType;
use Neos\Flow\Aop\JoinPointInterface;
use Neos\Flow\Annotations as Flow;
use Neos\Flow\ObjectManagement\ObjectManager;
use Neos\Flow\Reflection\ReflectionService;
use Newland\Toubiz\Sync\Neos\Domain\Model\Article;
use Newland\Toubiz\Sync\Neos\Domain\Repository\ArticleRepository;

/**
 * @Flow\Aspect()
 */
class TranslatePropertiesWhenCreatingNewLanguageVariants
{

    /**
     * @var ReflectionService
     * @Flow\Inject()
     */
    protected $reflectionService;

    /**
     * @var ObjectManager
     * @Flow\Inject()
     */
    protected $objectManager;

    /**
     * @param JoinPointInterface $joinPoint
     * @Flow\After("method(Neos\ContentRepository\Domain\Model\Node->createVariantForContext())")
     */
    public function onNewVariantCreation(JoinPointInterface $joinPoint): void
    {
        /** @var Node|null $oldNode */
        $oldNode = $joinPoint->getProxy();
        /** @var Node|null $newNode */
        $newNode = $joinPoint->getResult();

        if (!$oldNode || !$newNode) {
            return;
        }

        /** @var string|null $oldLanguage */
        $oldLanguage = $oldNode->getDimensions()['language'][0] ?? null;
        /** @var string|null $newLanguage */
        $newLanguage = $newNode->getDimensions()['language'][0] ?? null;

        if ($oldLanguage === null || $newLanguage === null || $oldLanguage === $newLanguage) {
            return;
        }

        $translators = $this->getApplicableTranslators($newNode->getNodeType(), $oldLanguage, $newLanguage);
        if (empty($translators)) {
            return;
        }

        $properties = (array) $newNode->getProperties();
        foreach ($translators as [ $translator, $method ]) {
            $properties = $translator->{$method}($properties, $newLanguage, $oldLanguage);
        }
        foreach ($newNode->getPropertyNames() as $propertyName) {
            $newNode->removeProperty($propertyName);
        }
        foreach ($properties as $name => $value) {
            $newNode->setProperty($name, $value);
        }
    }

    /**
     * @return mixed[][]
     */
    protected function getApplicableTranslators(
        NodeType $nodeType,
        string $sourceLanguage,
        string $targetLanguage
    ): array {
        $translators = [];
        $translatorClasses = $this->reflectionService->getClassesContainingMethodsAnnotatedWith(
            TranslatesNodeProperties::class
        );

        foreach ($translatorClasses as $class) {
            $methods = $this->reflectionService->getMethodsAnnotatedWith(
                $class,
                TranslatesNodeProperties::class
            );


            foreach ($methods as $method) {
                /** @var TranslatesNodeProperties $annotation */
                $annotation = $this->reflectionService->getMethodAnnotation(
                    $class,
                    $method,
                    TranslatesNodeProperties::class
                );

                if (!$this->nodeTypeMatchesOne($nodeType, $annotation->nodeTypes)) {
                    continue;
                }

                if ($annotation->sourceLanguages && !\in_array($sourceLanguage, $annotation->sourceLanguages)) {
                    continue;
                }

                if ($annotation->targetLanguages && !\in_array($targetLanguage, $annotation->targetLanguages)) {
                    continue;
                }

                $translators[] = [
                    $this->objectManager->get($class),
                    $method
                ];
            }
        }

        return $translators;
    }

    /** @param string[] $typesToCompare */
    private function nodeTypeMatchesOne(NodeType $type, array $typesToCompare): bool
    {
        foreach ($typesToCompare as $typeToCompare) {
            if ($type->isOfType($typeToCompare)) {
                return true;
            }
        }
        return false;
    }
}
