<?php
namespace Newland\Toubiz\Api\Service\Tportal\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithStarRatings;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;

/**
 * Lodging adapter.
 *
 * This represents an Article with mapping for the Tportal lodging.
 */
class CongressLocationAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithStarRatings
{
    use LanguageAware,
        WithCacheProperty,
        UsesFirstMediumAsMainMedium,
        StarRatingCommon;

    /** @var string|null */
    private $detailUriTemplate;

    public function setDetailUriTemplate(?string $detailUriTemplate): void
    {
        $this->detailUriTemplate = $detailUriTemplate;
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_CONGRESS_LOCATION;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['housedata']['out']['ID']);
    }

    public function getName(): string
    {
        $name = $this->object['housedata']['out']['address']['companyName1'] ?? null;
        if ($name === null) {
            $name = $this->object['housedata']['out']['describingName'];
        }

        $name = strip_tags($name);

        // Remove trailing dashes, commas and whitespace
        /** @var string $name */
        $name = \Safe\preg_replace('/[\s,-]*$/', '', $name);

        return $name;
    }

    public function getLatitude(): ?float
    {
        $latitude = $this->object['housedata']['out']['address']['gmapsLatitude'] ?? null;
        return $latitude ? ((float) $latitude) : null;
    }

    public function getLongitude(): ?float
    {
        $longitude = $this->object['housedata']['out']['address']['gmapsLongitude'] ?? null;
        return $longitude ? ((float) $longitude) : null;
    }

    public function getMainAddress(): AddressAdapterInterface
    {
        $address = $this->object;
        $address['name'] = $this->getName();
        return new CongressLocationAddressAdapter($address);
    }

    public function getDescription(): ?string
    {
        $part = $this->object['housedata']['out']['descriptions'];
        if ($part) {
            if (array_key_exists('description', $part)) {
                return $this->stringCleaner->purifyHtml((string) $part['description']);
            }

            $descriptionPart = $this->findInArray(
                $part,
                'descriptionTypeName',
                'Beschreibung'
            );

            if ($descriptionPart && !empty($descriptionPart['description'])) {
                return $this->stringCleaner->purifyHtml((string) $descriptionPart['description']);
            }
        }

        return null;
    }

    public function getMainCategory(): ?CategoryAdapterInterface
    {
        $categories = $this->getCategories();

        return $categories[0] ?? null;
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $categoryData = $this->object['housedata']['out']['touristicType'] ?? null;
        if ($categoryData === null) {
            return [];
        }

        return [
            new CategoryAdapter($categoryData),
        ];
    }


    /**
     * @return array
     */
    public function getAddresses(): array
    {
        return [];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache(
            'media',
            function () {
                if (!empty($this->object['housedata']['out']['multimediaURLs'])) {
                    foreach ($this->object['housedata']['out']['multimediaURLs'] as $item) {
                        /*
                         * Right now, only the preview image is being imported as there is
                         * no show view for lodgings.
                         *
                         * Next to that, $item can also be a non-array with useless
                         * information (like a string), or an array with an unexpected
                         * structure that does not provide a complete image.
                         */
                        if (is_array($item) && ($item['type']['value'] ?? null) === 'TP_Preview') {
                            return [ new MediumAdapter($item) ];
                        }
                    }
                }

                /*
                 * And for the case where there is no 'TP_Preview' despite that every
                 * record basically must have one, there is 'previewImageURL' but with
                 * an already cropped image.
                 */
                $url = $this->object['searchresult']['detailInfo']['previewImageURL'];
                if (!empty($item['url'])) {
                    $item = [
                        'ID' => md5($url),
                        'url' => $url,
                    ];
                    return [ new MediumAdapter($item) ];
                }

                return [];
            }
        );
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    /**
     * @return string|null
     */
    public function getDetailUri(): ?string
    {
        if ($this->detailUriTemplate === null || empty($this->object['housedata']['out']['niceUrl'])) {
            return null;
        }

        $niceUrl = ltrim($this->object['housedata']['out']['niceUrl'], '/');
        $this->detailUriTemplate = str_replace('{tportalUriSegment}', $niceUrl, $this->detailUriTemplate);
        $languageSegment = $this->getLanguage() ?? 'de';
        return str_replace('{languageSegment}', $languageSegment, $this->detailUriTemplate);
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    public function getAverageRating(): int
    {
        return (int) $this->object['searchresult']['serviceProviderRatingTotal']['averageGrade'] * 10;
    }

    public function getNumberOfRatings(): int
    {
        return (int) $this->object['searchresult']['serviceProviderRatingTotal']['numberOfRatings'];
    }

    private function parseAttributes(): array
    {
        return [];
    }

    public function getFiles(): array
    {
        return [];
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getSourceSystem(): string
    {
        return self::SOURCE_TPORTAL;
    }

    public function getSourceApiVersion(): ?string
    {
        return null;
    }

    public function getAdditionalSearchString(): ?string
    {
        return null;
    }
}
