<?php
namespace Newland\Toubiz\Api\Service\Meteotest\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\Weather\HourAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Weather\WeatherSymbol;

class HourAdapter implements HourAdapterInterface
{
    /**
     * @var \DateTime
     */
    private $date;
    /**
     * @var int
     */
    private $time = 0;
    /**
     * @var float
     */
    private $temperature = 0;
    /**
     * @var float
     */
    private $maximumTemperature;
    /**
     * @var float
     */
    private $minimumTemperature;
    /** @var string */
    private $symbol;

    /** @see https://meteotest.ch/en/weather-api/wetter-api-dokumentation/weather-symbols */
    private const SYMBOL_MAP = [
        '1' => WeatherSymbol::SUNNY,
        '2' => WeatherSymbol::MOSTLY_SUNNY,
        '3' => WeatherSymbol::CLOUDY,
        '4' => WeatherSymbol::HEAVY_CLOUDS,
        '5' => WeatherSymbol::THUNDERSTORM,
        '6' => WeatherSymbol::HEAVY_RAIN,
        '7' => WeatherSymbol::SNOW,
        '8' => WeatherSymbol::FOG,
        '9' => WeatherSymbol::SNOW,
        '10' => WeatherSymbol::RAIN,
        '11' => WeatherSymbol::RAIN,
        '12' => WeatherSymbol::SNOW,
        '13' => WeatherSymbol::THUNDERSTORM,
        '14' => WeatherSymbol::FOG,
        '15' => WeatherSymbol::SNOW,
    ];

    public function __construct(string $dateString, array $data)
    {
        $this->date = new \DateTime($dateString);
        $this->setTimeFromDateString($dateString);
        $this->parseData($data);
    }

    private function setTimeFromDateString(string $dateString): void
    {
        $date = new \DateTime($dateString);
        $this->time = (int) $date->format('G');
    }

    private function parseData(array $data): void
    {
        $this->temperature = round($data['tt']);
        if (isset($data['tx'])) {
            $this->maximumTemperature = $data['tx'];
        }
        if (isset($data['tn'])) {
            $this->minimumTemperature = $data['tn'];
        }
        $this->symbol = self::SYMBOL_MAP[(string) $data['sy']] ?? WeatherSymbol::UNKNOWN;
    }

    public function getTemperature(): float
    {
        return $this->temperature;
    }

    /** @see https://meteotest.ch/en/weather-api/wetter-api-dokumentation/weather-symbols */
    public function getSymbol(): string
    {
        return $this->symbol;
    }

    public function getExternalId(): string
    {
        return '';
    }

    /**
     * @return \DateTime
     */
    public function getDate(): \DateTime
    {
        return $this->date;
    }

    /**
     * The hour of the time in 0 - 23 format.
     *
     * @return int
     */
    public function getTime(): int
    {
        return $this->time;
    }

    public function getMinimumTemperature(): float
    {
        return $this->minimumTemperature ?? $this->temperature;
    }

    public function getMaximumTemperature(): float
    {
        return $this->maximumTemperature ?? $this->temperature;
    }
}
