<?php
namespace Newland\Toubiz\Api\Service\Sisag\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\RegionAdapterInterface;

/**
 * Region object adapter.
 *
 * Describes a region containing slopes and transportation facilities.
 */
class RegionAdapter extends AbstractObjectAdapter implements RegionAdapterInterface
{
    /**
     * @var \SimpleXMLElement
     */
    protected $xml;

    /**
     * Sets the full xml.
     *
     * This is necessary in order to receive all
     * dependend objects.
     *
     * @param \SimpleXMLElement $xml
     * @return void
     */
    public function setFullXml($xml)
    {
        $this->xml = $xml;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return (string) $this->object['id'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return (string) $this->object->NAME;
    }

    /**
     * @return array
     */
    public function getTransportationFacilities()
    {
        $items = [];
        foreach ($this->xml->LIFTE->children() as $item) {
            if ((string) $item->REGION['id'] !== $this->getExternalId()) {
                continue;
            }

            /*
             * A status of "0" indicates an inactive / non-visible
             * facility which should not be imported at all.
             */
            if ((int) $item->STATUS === 0) {
                continue;
            }

            $this->injectUpdatedAt($item);
            $items[] = (new TransportationFacilityAdapter($item));
        }
        return $items;
    }

    /**
     * @return array
     */
    public function getSlopes()
    {
        $items = [];
        foreach ($this->xml->PISTEN->children() as $item) {
            if ((string) $item->REGION['id'] !== $this->getExternalId()) {
                continue;
            }

            $this->injectUpdatedAt($item);
            $items[] = (new SlopeAdapter($item));
        }
        return $items;
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $attributes = [];

        if ($this->xml->AVALANCHE) {
            foreach ($this->xml->AVALANCHE->ROUTE as $item) {
                if ((string) $item->REGION['id'] == $this->getExternalId()) {
                    $attributes['avalancheStatus'] = (int) $item->STATUS;
                    break;
                }
            }
        }

        if ($this->xml->METEO) {
            foreach ($this->xml->METEO->ROUTE as $item) {
                if ((string) $item->REGION['id'] == $this->getExternalId()) {
                    $attributes['newSnowInLast24Hours'] = (int) $item->NEUSCHNEE24H;
                    $attributes['snowHeight'] = (int) $item->SCHNEEHOEHE;
                    $attributes['lastSnowfall'] = (string) $item->LSCHNEEFALL;
                    break;
                }
            }
        }

        return $attributes;
    }

    /**
     * @return \DateTime
     */
    public function getUpdatedAt()
    {
        return (new \DateTime((string) $this->xml->AUSGABE));
    }

    /**
     * Adds a child node 'AUSGABE' to the given xml node
     * which represents the node from the parent object containing
     * the updated-at timestamp.
     *
     * @param \SimpleXMLElement $xmlNode
     * @return void
     */
    protected function injectUpdatedAt(\SimpleXMLElement &$xmlNode)
    {
        $xmlNode->addChild('AUSGABE', $this->xml->AUSGABE);
    }
}
