<?php
namespace Newland\Toubiz\Api\Service\Sisag;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Sisag\ObjectAdapter\RegionAdapter;

/**
 * Service for SISAG API.
 *
 * Concrete implementation to communicate with
 * the SISAG API.
 */
class ApiService extends AbstractService
{
    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'http://siscontrol.sisag.ch/sismedia/:clientName:/SisControlAktuell.xml';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [];

    /**
     * @var array Internal cache.
     */
    protected $cache = [];

    /**
     * Fetch regions.
     *
     * Each part of the dataset is split up into their own child node.
     * Due to the nature of importers and the database structure,
     * each region gets assigned the other corresponding xml nodes.
     *
     * @return array<RegionAdapter>
     */
    public function fetchRegions()
    {
        $data = $this->fetchRawReport();
        $collection = [];
        foreach ($data->REGIONS->children() as $item) {
            $region = (new RegionAdapter($item));
            $region->setFullXml($data);

            $collection[] = $region;
        }
        return $collection;
    }

    /**
     * Fetch tagestipp.
     *
     * @return null|string
     */
    public function fetchAdditionalInformation()
    {
        $data = $this->fetchRawReport();

        if (isset($data->REPORT->TAGESTIPP)) {
            return $data->REPORT->TAGESTIPP;
        }

        return null;
    }

    /**
     * Fetch report.
     *
     * The "API" basically is an XML file that is being generated
     * in regular intervals.
     *
     * @return \SimpleXMLElement
     * @throws \Exception if no data has been received from the API
     */
    public function fetchRawReport()
    {
        if (array_key_exists('fetchRawReport', $this->cache)) {
            return $this->cache['fetchRawReport'];
        }

        // Override already initialized http client.
        $baseUri = str_replace(':clientName:', $this->clientName, self::DEFAULT_BASE_URI);
        $this->httpClient = new \GuzzleHttp\Client();
        $response = $this->httpClient->request('GET', $baseUri);

        if ($response->getStatusCode() == 200) {
            $this->cache['fetchRawReport'] = simplexml_load_string($response->getBody());
            return $this->cache['fetchRawReport'];
        }

        throw new \Exception('No data received for synchronizing!');
    }
}
