<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\DirectMarketerAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\AttributeAdapter;

/**
 * Direct marketer adapter.
 *
 * This represents an Article with mapping for the Toubiz specific direct marketer.
 */
class DirectMarketerAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    /**
     * @return int
     */
    public function getMainType()
    {
        return ArticleConstants::TYPE_DIRECT_MARKETER;
    }

    /**
     * @return string
     */
    public function getExternalId()
    {
        return $this->object['remote_id'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->object['name'];
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        $description = null;
        $value = '';
        if (!is_array($this->object['description'])) {
            return $description;
        }
        if (array_key_exists('direkt_event', $this->object['description'])) {
            $value .= $this->object['description']['direkt_event']['text'];
        }
        if (array_key_exists('direkt', $this->object['description'])) {
            $value .= $this->object['description']['direkt']['text'];
        }


        if (!empty($value)) {
            $description = $value;
        }

        return $description;
    }

    /**
     * @return array<AddressAdapterInterface>
     */
    public function getAddresses()
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress()
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return array<CategoryAdapter>
     */
    public function getCategories()
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia()
    {
        if (!is_array($this->object['images'])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $media[] = (new MediumAdapter($image));
        }
        return $media;
    }

    public function hasAttributes()
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $attributes = $this->parseAttributes();

        $items = [];

        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    /**
     * @return string
     */
    public function getFacebookUri()
    {

        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'facebook'
        );

        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getTwitterUri()
    {
        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'twitter'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getInstagramUri()
    {
        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'instagram'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getYoutubeUri()
    {
        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'youtube'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getWikipediaUri()
    {
        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'wikipedia'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getFlickrUri()
    {
        if (!array_key_exists('social_media', $this->object)) {
            return '';
        }

        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'flickr'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return array
     */
    public function getFiles()
    {
        return [];
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return $this->object['headline_quellmandant'];
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        $author = null;
        if (!empty($this->object['author'])) {
            $author = $this->object['author'];
        }

        return $author;
    }

    /**
     * @return string
     */
    public function getBookingUri()
    {
        return '';
    }

    /**
     * @return string
     */
    public function getDetailUri()
    {
        return '';
    }

    /**
     * @return bool
     */
    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    /**
     * The complete opening times data.
     *
     * @return string
     */
    public function getOpeningTimes()
    {
        return (string) json_encode($this->object['opentimes']);
    }

    /**
     * Return the star classification for hotels.
     *
     * @return int
     */
    public function getStarClassification()
    {
        return 0;
    }

    /**
     * Return the average rating on a scale of 0-100
     *
     * @return int
     */
    public function getAverageRating()
    {
        return 0;
    }

    /**
     * Return the number of ratings.
     *
     * @return int
     */
    public function getNumberOfRatings()
    {
        return 0;
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $keys = [
            'dv_marketing' => DirectMarketerAttributes::SHOPPING,
            'dv_prod_cat_aufstriche' => DirectMarketerAttributes::OIL,
            'dv_prod_cat_gemuese_obst' => DirectMarketerAttributes::VEGETABLES,
            'dv_prod_cat_getraenke' => DirectMarketerAttributes::DRINKS,
            'dv_prod_cat_saison' => DirectMarketerAttributes::SEASONALS,
            'dv_special' => DirectMarketerAttributes::COMMODITIES,
            'dv_prod_cat_fleisch' => DirectMarketerAttributes::MEATS,
            'dv_prod_cat_gefluegel_fisch' => DirectMarketerAttributes::FISH,
            'dv_prod_cat_milch' => DirectMarketerAttributes::MILK,
            'dv_prod_cat_getreide' => DirectMarketerAttributes::CEREALS,
            'dv_prod_cat_handwerk' => DirectMarketerAttributes::CRAFTS,
            'dv_decoration' => DirectMarketerAttributes::AWARDS,
            'dv_decoration_regional' => DirectMarketerAttributes::REGIONAL_AWARDS,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        return $attributes;
    }

    /**
     * Extract attributes from a volatile set of data.
     *
     * @param string $key
     * @return array|null
     */
    private function extractAttributes($key)
    {
        $items = [];

        // The desired key can be in two different places.
        if (array_key_exists($key, $this->object)) {
            $attributes = $this->object[$key];
        } elseif (is_array($this->object['property'])
            && array_key_exists($key, $this->object['property'])
        ) {
            $attributes = $this->object['property'][$key];
        } elseif (is_array($this->object['classification'])
            && array_key_exists($key, $this->object['classification'])
        ) {
            $attributes = $this->object['classification'][$key];
        } else {
            return null;
        }

        // Also, the structure is different for finding the attributes.
        if (is_array($attributes[0])
            && array_key_exists('properties', $attributes[0])
        ) {
            $attributes = $attributes[0]['properties'];
        }

        foreach ($attributes as $entry) {
            // For the direct Marketers Value is never used with exception for the awards
            $items[] = $entry['text'];
        }

        if (!empty($items)) {
            return $items;
        }
    }
}
