<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Tests\Unit\Guzzle;

use GuzzleHttp\Promise\FulfilledPromise;
use GuzzleHttp\Promise\RejectedPromise;
use Newland\Toubiz\Api\Exception\InvalidReturnValueException;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use PHPUnit\Framework\TestCase;

class ConcurrentPaginatedRequestsTest extends TestCase
{

    public function testCallsCallbackWithPageNumber(): void
    {
        $fulfilled = [];
        $rejected = [];
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) use (&$fulfilled, &$rejected) {
            if ($page < 9) {
                $fulfilled[] = $page;
                return new FulfilledPromise(null);
            }

            $rejected[] = $page;
            return new RejectedPromise(null);
        });

        $pool->start()->wait();

        $this->assertEquals([ 1, 2, 3, 4, 5, 6, 7, 8 ], $fulfilled);

        // Because we start 3 requests concurrently there will be up to 3 rejected promises
        // as they overshoot the maximum. This is expected.
        $this->assertEquals([ 9, 10, 11 ], $rejected);
    }

    public function testRespectsMaximumPageNumberPassedEvenIfPromisesKeepOnBeingFulfilled(): void
    {
        $calls = [];
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) use (&$calls) {
            $calls[] = $page;
            return new FulfilledPromise(null);
        });

        $pool->setLastPage(6);
        $pool->start()->wait();

        $this->assertEquals([ 1, 2, 3, 4, 5, 6 ], $calls);
    }

    public function testItAbortsExecutionToPreventIfiniteLoops()
    {
        $lastPage = 0;
        (new ConcurrentPaginatedRequests(3, function(int $page) use (&$lastPage) {
            $lastPage = $page;
            return new FulfilledPromise(null);
        }))->start()->wait();

        $this->assertEquals($lastPage, 9999);
    }

    public function testThrowsExceptionIfHandlerDoesNotReturnPromise()
    {
        $this->expectException(InvalidReturnValueException::class);
        $this->expectExceptionMessageRegExp('/Promise/');
        (new ConcurrentPaginatedRequests(3, function() {
            return null;
        }))->start()->wait();
    }

    public function testResolvesPromiseWithArrayOfResults(): void
    {
        $pool = new ConcurrentPaginatedRequests(3, function(int $page) {
            if ($page < 9) {
                return new FulfilledPromise($page);
            }

            return new RejectedPromise(null);
        });

        $this->assertEquals(
            [ 1, 2, 3, 4, 5, 6, 7, 8 ],
            $pool->start()->wait(true)
        );
    }

    public function testRejectsPromiseIfExceptionThrown(): void
    {
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('This is a test exception');
        (new ConcurrentPaginatedRequests(3, function() {
            throw new \Exception('This is a test exception');
        }))->start()->wait();
    }

}
