<?php
namespace Newland\Toubiz\Api\Service\Tportal\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\PackageAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Utility\UrlConverter;
use function Safe\preg_match_all;

class PackageAdapter extends AbstractObjectAdapter implements PackageAdapterInterface
{
    use LanguageAware;

    /** @var null|string */
    private $detailUriTemplate;

    public function setDetailUriTemplate(?string $detailUriTemplate): void
    {
        $this->detailUriTemplate = $detailUriTemplate;
    }

    /**
     * Returns the external id used by the API
     * to identify the record.
     *
     * @return string
     */
    public function getExternalId(): string
    {
        return strip_tags((string) ($this->object['package']['packageID'] ?? ''));
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return strip_tags((string) ($this->object['package']['packageName'] ?? ''));
    }

    /**
     * @return string|null
     */
    public function getPreviewImage(): ?string
    {
        foreach ($this->object['serviceData']['multimediaURLs'] ?? [] as $mediaItem) {
            $type = $mediaItem['type']['value'] ?? null;
            $url = $mediaItem['url'] ?? null;

            if ($type === 'Preview' && $url !== null) {
                return UrlConverter::forceHttps($url);
            }
        }

        return UrlConverter::forceHttps($this->object['package']['previewImageURL'] ?? null);
    }

    /**
     * @return string[]
     */
    public function getBenefits(): array
    {
        $description = $this->getDescription('DESCRIPTION_INCLUDED_SERVICES');

        if ($description === null) {
            return [];
        }

        preg_match_all('/\<li\>(.*?)\<\/li\>/', $description, $matches);
        return array_map(
            function (string $item) {
                return trim(strip_tags($item));
            },
            $matches[1] ?? []
        );
    }

    /**
     * @return float|null
     */
    public function getMinimumNights()
    {
        return $this->object['package']['minimumNights'] ?? null;
    }

    /**
     * @return float|null
     */
    public function getStartingPrice()
    {
        $group = $this->getFeatureGroup('packageprice');
        if ($group) {
            return $group['features']['number'] ?? null;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getStartingPriceCurrency()
    {
        return 'EUR';
    }

    private function getFeatureGroup(string $externalId)
    {
        foreach ($this->object['serviceData']['featureGroups'] ?? [] as $featureGroup) {
            $featureExternalId = $featureGroup['features']['value']['externalID'] ?? null;
            if ($externalId === $featureExternalId) {
                return $featureGroup;
            }
        }
        return null;
    }

    private function getDescription(string $externalId): ?string
    {
        foreach ($this->object['serviceData']['descriptions'] ?? [] as $description) {
            $descriptionExternalId = $description['externalID'] ?? null;
            if ($externalId === $descriptionExternalId) {
                return $description['description'];
            }
        }
        return null;
    }

    public function getDetailUri(): ?string
    {
        if (!$this->detailUriTemplate) {
            return null;
        }

        $niceUrl = $this->object['serviceData']['niceUrl'] ?? null;
        if (!$niceUrl) {
            return null;
        }

        $niceUrl = ltrim($niceUrl, '/');
        $this->detailUriTemplate = str_replace('{tportalUriSegment}', $niceUrl, $this->detailUriTemplate);
        $languageSegment = $this->getLanguage() ?? 'de';

        return str_replace('{languageSegment}', $languageSegment, $this->detailUriTemplate);
    }
}
