<?php
namespace Newland\Toubiz\Api\Utility;

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectionException;
use Psr\Log\LoggerAwareTrait;
use function Safe\json_encode;

class RetryPool
{
    use LoggerAwareTrait;

    const SECOND_TO_MICROSECONDS = 1000000;

    /** @var int */
    private $maxRetries;

    /** @var float */
    private $sleepSecondsBeforeRetry;

    /** @var \Throwable */
    private $lastException;

    /** @var array|null */
    private $debuggingInformation;

    public function __construct(int $maxRetries, float $sleepSecondsBeforeRetry = 3.0)
    {
        $this->maxRetries = $maxRetries;
        $this->sleepSecondsBeforeRetry = $sleepSecondsBeforeRetry;
        $this->forceSleepTimeToZeroIfInTestingEnvironment();
    }

    public function retryOnException(callable $block)
    {
        $retries = 0;

        do {
            try {
                return $block();
            } catch (\Exception $e) {
                $this->incrementRetriesAndThrowIfExceedsLimit($e, $retries);
            }
        } while ($retries < $this->maxRetries);

        throw $this->maximumRetryException();
    }

    public function retryOnPromiseRejection(callable $block): PromiseInterface
    {
        return $this->doRetryOnPromiseRejection($block);
    }

    public function setDebuggingInformation(array $information): void
    {
        $this->debuggingInformation = $information;
    }

    private function doRetryOnPromiseRejection(callable $block, int &$currentRetryCount = 0): PromiseInterface
    {
        /** @var PromiseInterface $promise */
        $promise = $block();
        return $promise->otherwise(function ($e) use (&$block, &$currentRetryCount) {
            if ($e instanceof RetryAbortingError) {
                $previous = $e->getPrevious();
                if ($previous !== null) {
                    throw $previous;
                }
            }
            $this->incrementRetriesAndThrowIfExceedsLimit($e, $currentRetryCount);
            return $this->doRetryOnPromiseRejection($block, $currentRetryCount);
        });
    }

    private static $i = 0;

    /**
     * @param \Throwable|string $e
     */
    private function incrementRetriesAndThrowIfExceedsLimit($e, int &$retries): void
    {
        if (\is_string($e)) {
            $e = new RejectionException($e);
        }

        $this->lastException = $e;
        if (++$retries >= $this->maxRetries) {
            throw $this->maximumRetryException();
        }

        $this->logger->warning(sprintf(
            "Action failed (Retry #%d): %s. Retrying in %d seconds. \n %s",
            $retries,
            $e->getMessage(),
            $this->sleepSecondsBeforeRetry,
            $this->debuggingInformation()
        ));
        usleep((int) ($this->sleepSecondsBeforeRetry * static::SECOND_TO_MICROSECONDS));
    }

    private function maximumRetryException(): MaximumRetriesExceededException
    {
        return new MaximumRetriesExceededException(
            'Maximum retries exceeded ' . $this->debuggingInformation(),
            0,
            $this->lastException
        );
    }

    private function debuggingInformation(): string
    {
        if (!$this->debuggingInformation) {
            return '';
        }

        return json_encode($this->debuggingInformation);
    }

    /**
     * Helper for testing environments: Leaving the default 3.0 seconds sleep time
     * makes tests seem sluggish if they fail.
     */
    private function forceSleepTimeToZeroIfInTestingEnvironment(): void
    {
        if (getenv('FLOW_CONTEXT') === 'Testing') {
            $this->sleepSecondsBeforeRetry = 0.0;
        }
    }
}
