<?php declare(strict_types=1);

namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Service\Toubiz\Legacy\TourApiService;

use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\TourApiService\TourAdapter;
use PHPUnit\Framework\TestCase;
use Psr\Log\Test\TestLogger;

class TourAdapterTest extends TestCase
{

    /** @dataProvider provideValidGpxFiles */
    public function testGetsGeometryFromGpxFile(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $attributeNames = array_map(function(AttributeAdapterInterface $attribute) {
            return $attribute->getName();
        }, $attributes);
        $this->assertContains(TourAttributes::GEOMETRY, $attributeNames, 'Attributes should include geometry');
    }

    /** @dataProvider provideErroneousGpxFiles */
    public function testErrorWhenParsingGpxFileDoesNotCompletelyAbortExecution(string $file): void
    {
        [ $tour, $logger ] = $this->adapterForGpsFile($file);
        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);

        $messages = array_map(function(array $record) {
            return $record['message'];
        }, $logger->records);
        $this->assertContains('Error parsing GPS File', implode('|', $messages));
    }

    /** @dataProvider provideInvalidGpxFiles */
    public function testInvalidOrEmptyGpxFilesDoNotAddGeometry(string $file): void
    {
        [ $tour ] = $this->adapterForGpsFile($file);

        $attributes = $tour->getAttributes();
        $this->assertArrayNotHasKey(TourAttributes::GEOMETRY, $attributes);
    }

    public function provideErroneousGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/erroneous-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideValidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/valid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return $data;
    }

    public function provideInvalidGpxFiles(): array
    {
        $data = [];
        $path = __DIR__ . '/../../../../../Fixtures/gpx/invalid-gpx-file__';
        foreach (glob($path . '*.gpx') as $file) {
            $name = str_replace([ $path, '.gpx' ], '', $file);
            $data[$name] = [ $file ];
        }
        return array_merge($data, $this->provideErroneousGpxFiles());
    }

    private function adapterForGpsFile(string $file): array
    {
        $tour = new TourAdapter([
            'remote_id' => 'TEST__' . preg_replace('/\W/', '-', $file),
            'object' => [
                'data_map' => [
                    'geodata_file_gpx' => [
                        'content' => [
                            'path' => $file
                        ]
                    ]
                ]
            ]
        ]);

        $logger = new TestLogger();
        $tour->setLogger($logger);

        return [ $tour, $logger ];
    }
}
