<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\Regex;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface
{
    use LanguageAware,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    public function getDescription(): ?string
    {
        return $this->object['description']['gastro']['text']
            ?? $this->object['description']['gastro_event']['text']
            ?? $this->getDataMapString('description');
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache(
            'media',
            function () {
                if (!is_array($this->object['images'])) {
                    return [];
                }

                $items = [];

                if (ArrayUtility::isAssociative($this->object['images'])) {
                    foreach ($this->object['images'] as $imageCollection) {
                        if (is_array($imageCollection)) {
                            foreach ($imageCollection as $imageToAdd) {
                                $items[] = $imageToAdd;
                            }
                        } else {
                            $items[] = $imageCollection;
                        }
                    }
                } else {
                    foreach ($this->object['images'] as $imageCollection) {
                        $items[] = $imageCollection;
                    }
                }

                $media = [];
                foreach ($items as $image) {
                    $media[] = new MediumAdapter($image);
                }

                return $media;
            }
        );
    }

    /**
     * @return AttributeAdapterInterface[]
     */
    public function getAttributes(): array
    {
        $attributes = $this->parseAttributes();

        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $attributes,
            function (string $name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    /** @return FileAdapter[] */
    public function getFiles(): array
    {
        if (!is_array($this->object['files'])) {
            return [];
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    public function getBookingUri(): ?string
    {
        if (!is_array($this->object['property'])
            || !array_key_exists('gastro_lunchgate', $this->object['property'])
            || !is_array($this->object['property']['gastro_lunchgate'])
        ) {
            return null;
        }

        $value = $this->object['property']['gastro_lunchgate'][0]['properties'][0]['value'];

        // This is wrong data due to missing validations.
        if (strpos($value, '@') !== false) {
            return null;
        }

        return $value;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    private function extractAttributes(string $key, bool $valueOnly = false): ?array
    {
        $attributes = $this->object[$key]
            ?? $this->object['property'][$key]
            ?? [];

        $normalizedAttributes = [];
        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if (array_key_exists('properties', $attribute)) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) use ($valueOnly) {
                if ($valueOnly) {
                    return $item['value'];
                }

                 // Most of the time, text is only used. But if it also has a value given, it needs to be concatenated.
                if (!empty($item['value'])) {
                    return $item['text'] . ': ' . $item['value'];
                }

                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    private function extractDecorationAttributes($key): ?array
    {
        $attributes = $this->object[$key]
            ?? $this->object['decoration'][$key]
            ?? [];

        $normalizedAttributes = [];

        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }

            if (array_key_exists('properties', $attribute)) {
                if (is_array($attribute['properties']) && $attribute['properties'] !== null) {
                    foreach ($attribute['properties'] as $property) {
                        $normalizedAttributes[] = $property;
                    }
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) {
                // Most of the time, text is only used. But if it also has a value given, it needs to be concatenated.
                if (!empty($item['value'])) {
                    return Regex::replace('/\#\#\#/', ' - ', $item['text']);
                }

                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $translated = [
            'gastro_ambient' => GastronomyAttributes::AMBIANCE,
            'gastro_group' => GastronomyAttributes::GROUPS,
            'gastro_kitchen' => GastronomyAttributes::KITCHEN_STYLE,
            'gastro_style' => GastronomyAttributes::GASTRONOMY_STYLE,
            'gastro_person' => GastronomyAttributes::CAPACITY,
            'gastro_payment' => GastronomyAttributes::ACCEPTED_PAYMENTS,
            'gastro_pricesegment' => GastronomyAttributes::PRICE_SEGMENT,
            'gastro_property' => GastronomyAttributes::SERVICES,
            'gastro_language' => GastronomyAttributes::SPOKEN_LANGUAGES,
            'gastro_diet' => GastronomyAttributes::HABITS_ALLERGIES,
        ];
        $untranslated = [
            'child' => GastronomyAttributes::FAMILIES,
        ];
        $decorationKeys = [
            'house_decoration' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_aral' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_dehoga' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_fine' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_michelin' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_millau' => GastronomyAttributes::HOUSE_DECORATION,
            'house_decoration_varta' => GastronomyAttributes::HOUSE_DECORATION,
        ];

        foreach ($translated as $key => $name) {
            $attribute = $this->extractAttributes($key, false);
            if (!empty($attribute)) {
                $attributes[$name] = $attribute;
            }
        }

        foreach ($untranslated as $key => $name) {
            $attribute = $this->extractAttributes($key, true);
            if (!empty($attribute)) {
                $attributes[$name] = $attribute;
            }
        }

        foreach ($decorationKeys as $key => $name) {
            $attribute = $this->extractDecorationAttributes($key);
            if (!empty($attribute)) {
                $attributes[$name] = $attribute;
            }
        }


        return $attributes;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getLocationId(): ?string
    {
        $cityName = $this->object['id_location'] ?? null;

        if ($cityName !== null) {
            return 'location_' . $cityName;
        }
        return null;
    }
}
