<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\CityApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleTypeCityAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\CityAttributes;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\FileAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\RelatedLists\RelatedListsAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Service\WithUuidPredictionService;
use Newland\Toubiz\Api\Service\WithUuidPredictionServiceInterface;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;

class CityAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleTypeCityAdapterInterface,
    WithUuidPredictionServiceInterface,
    ArticleWithSocialMediaLinksAdapterInterface
{
    use LanguageAware;
    use WithUuidPredictionService;
    use WithCacheProperty;
    use UsesFirstMediumAsMainMedium;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_CITY;
    }

    public function getExternalId(): string
    {
        return $this->object['remote_id'];
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getAbstract(): ?string
    {
        return $this->object['object']['data_map']['intro']['content'] ?? null;
    }

    public function getDescription(): ?string
    {
        return $this->object['object']['data_map']['description']['content'] ?? null;
    }

    public function getLatitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[0];
        }

        return null;
    }

    public function getLongitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[1];
        }

        return null;
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        return [];
    }

    /**
     * @return MediumAdapterInterface[]
     */
    public function getMedia(): array
    {
        return $this->cache(
            'media',
            function () {
                $media = [];
                if (!is_array($this->object['object']['data_map']['gallery']['content'])) {
                    return $media;
                }

                foreach ($this->object['object']['data_map']['gallery']['content'] as $key => $image) {
                    // Skip importing media records without a source uri.
                    if (!empty($image['reference'])) {
                        $media[] = new MediumAdapter($image);
                    }
                }

                return $media;
            }
        );
    }

    /**
     * @return FileAdapterInterface[]
     */
    public function getFiles(): array
    {
        return [];
    }

    /**
     * TODO Remove method. The same data can be retrieved form `getAttributes`
     *
     * @return bool
     * @deprecated
     */
    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return AttributeAdapterInterface[]
     */
    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialUri('flickr');
    }

    private function getSocialUri(string $dataMapItemName): ?string
    {
        $value = $this->object['object']['data_map'][$dataMapItemName]['content'] ?? null;

        if (!$this->validateUrl($value)) {
            return null;
        }

        return (string) $value;
    }

    private function validateUrl(?string $url): ?bool
    {
        if ($url === null) {
            return null;
        }
        return (bool) preg_match(
            '/^(?:http(s)?:\/\/)?[\w.-]+(?:\.[\w\.-]+)+[\w\-\._~:\/?#[\]@!\$&\(\)\*\+,;=.]+$/',
            $url
        );
    }

    public function getSourceName(): ?string
    {
        return null;
    }

    public function getAuthorName(): ?string
    {
        return $this->object['object']['author'] ?? null;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getRelatedLists(): ?RelatedListsAdapterInterface
    {
        $adapter = new RelatedListsAdapter($this->object);
        $adapter->setUuidPredictionService($this->uuidPredictionService);

        return $adapter;
    }

    private function parseAttributes(): array
    {
        $attributes = [];

        $childrenList = $this->getChildrenList();
        foreach ($this->extensionAttributeMapping as $extension => $mapping) {
            foreach ($mapping as $normalized => $external) {
                if (!empty($childrenList[$extension]['object']['data_map'][$external]['content'])) {
                    $attributes[$normalized] = $childrenList[$extension]['object']['data_map'][$external]['content'];
                }
            }
        }

        $attributes = array_merge($attributes, $this->parseTags());

        return $attributes;
    }

    private function parseTags(): array
    {
        $attributes = [];

        $tagValues = ArrayUtility::trimExplode(',', $this->object['object']['data_map']['tags']['content'] ?? '');
        if (!empty($tagValues)) {
            $attributes[CityAttributes::TAG] = $tagValues;
        }

        return $attributes;
    }

    private function getChildrenList(): array
    {
        $children = [];
        foreach ($this->object['object_children_list'] ?? [] as $child) {
            foreach ($child as $name => $content) {
                $children[$name] = $content;
            }
        }
        return $children;
    }

    private $extensionAttributeMapping = [
        'kurort_extension' => [
            CityAttributes::CURE_DESCRIPTION => 'cure_description',
            CityAttributes::CURE_REGION => 'cure_region',
            CityAttributes::CURE_ALTITUDE => 'cure_lage',
            CityAttributes::CURE_CLIMATE => 'cure_klima',
            CityAttributes::CURE_AILMENTS_TAG => 'cure_heilanzeigen_tag',
            CityAttributes::CURE_AILMENTS => 'cure_heilanzeigen',
            CityAttributes::CURE_CURES => 'cure_heilmittel',
            CityAttributes::CURE_SCREENING => 'cure_vorsorge',
            CityAttributes::CURE_COMPACT_CURE => 'cure_kompaktkur',
            CityAttributes::CURE_THERAPY => 'cure_therapie',
            CityAttributes::CURE_BATHS => 'cure_baeder',
            CityAttributes::CURE_MASSAGE => 'cure_massage',
            CityAttributes::CURE_WHOLESOME => 'cure_ganzheitlich',
            CityAttributes::CURE_VARIOUS => 'cure_sonstiges',
            CityAttributes::CURE_LABEL => 'cure_praedikat',
        ],
    ];

    /**
     * Cities have a list of zip codes. This is important for filtering by city.
     *
     * @return string[]
     */
    public function getZipCodes(): array
    {
        return explode(',', $this->object['object']['data_map']['zip_all']['content']);
    }

    /**
     * The id from the toubiz system. Needed for filtering of tportal event lists.
     *
     * @return string
     */
    public function getIdToubiz(): string
    {
        return $this->object['object']['data_map']['toubiz_id']['content'] ?? '';
    }

    /**
     * The id from the tomas system. Needed for filtering of tportal lodging lists.
     *
     * @return string
     */
    public function getIdTomas(): string
    {
        return $this->object['object']['data_map']['tomas_id']['content'] ?? '';
    }

    public function getClaim(): string
    {
        return $this->object['object']['data_map']['claim']['content'] ?? '';
    }

    public function getFacts(): string
    {
        return $this->object['object']['data_map']['facts']['content'] ?? '';
    }

    public function getWebcamUrl(): ?string
    {
        $url = $this->object['object']['data_map']['webcam']['content'] ?? '';
        // The API also returns 'http://' or 'https://' as address, which should be blank.
        if (strlen($url) <= 8) {
            return null;
        }

        // The protocol may also be missing.
        if (strpos($url, 'http') !== 0) {
            $url = 'http://' . $url;
        }

        return $url;
    }

    public function getWebcamDescription(): ?string
    {
        return $this->object['object']['data_map']['webcam']['description'] ?? '';
    }

    public function getNews(): ?string
    {
        return $this->object['object']['data_map']['news']['content'] ?? '';
    }
}
