<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\MonthConstants;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\TourAttributes;
use Newland\Toubiz\Api\ObjectAdapter\FileAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\WithCacheProperty;

/**
 * Tour adapter.
 *
 * Describes a tour.
 */
class TourAdapter extends AbstractObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;
    use WithCacheProperty;

    /** @var array|null */
    private $cachedGeometry;

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE . '_' . $this->object->tour['id'];
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_TOUR;
    }

    public function getName(): string
    {
        return (string) $this->object->tour->title;
    }

    public function getAbstract(): ?string
    {
        $value = (string) $this->object->tour->shortText;
        return $value ?: null;
    }

    public function getDescription(): ?string
    {
        $value = (string) $this->object->tour->longText;
        return $value ?: null;
    }

    public function getLatitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][0];
        }

        return null;
    }

    public function getLongitude(): ?float
    {
        $geometry = $this->extractGeometry();
        if (\count($geometry) > 0) {
            return $geometry[0][1];
        }

        return null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        $geometry = $this->extractGeometry();
        $object = $this->object;
        if (\count($geometry) > 0) {
            $object->firstGeometry->lat = $geometry[0][0];
            $object->firstGeometry->lon = $geometry[0][1];
        }
        return new AddressAdapter($object);
    }

    /**
     * @return CategoryAdapterInterface[]
     */
    public function getCategories(): array
    {
        return [
            new CategoryAdapter($this->object->tour->category),
        ];
    }

    /**
     * @return MediumAdapterInterface[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            $media = [];
            if ($this->object->tour->images->image) {
                foreach ($this->object->tour->images->image as $image) {
                    $media[] = new MediumAdapter($image);
                }
            }

            return $media;
        });
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        $primaryImageId = TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE . '_' . $this->object->tour->primaryImage['id'];
        $images = $this->getMedia();

        if (empty($images)) {
            return null;
        }

        // Set the first image as the default main image.
        $primaryImage = $images[0];

        foreach ($images as $image) {
            if ($image->getExternalId() === $primaryImageId) {
                $primaryImage = $image;
                break;
            }
        }

        return $primaryImage;
    }

    /**
     * @return FileAdapterInterface[]
     */
    public function getFiles(): array
    {
        return [];
    }

    /**
     * @return bool
     */
    public function hasAttributes(): bool
    {
        return true;
    }

    /**
     * @return array
     */
    public function getAttributes(): array
    {
        $items = [
            new AttributeAdapter($this->getExternalId(), 'geometry', $this->extractGeometry()),
        ];

        if ($this->object->tour->properties->property) {
            foreach ($this->object->tour->properties->property as $property) {
                $items[] = new AttributeAdapter(
                    $this->getExternalId(),
                    'properties',
                    (string) $property['tag']
                );
            }
        }

        if ($this->object->tour->labels && $this->object->tour->labels->attributes()) {
            foreach ($this->object->tour->labels->attributes() as $attribute => $value) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'labels', $attribute);
            }
        }

        if ($this->object->tour->crossCountryTechnique) {
            foreach ($this->object->tour->crossCountryTechnique->property as $technique) {
                $items[] = new AttributeAdapter(
                    $this->getExternalId(),
                    'crossCountryTechnique',
                    (string) $technique['tag']
                );
            }
        }

        $months = [
            'jan' => MonthConstants::JANUARY,
            'feb' => MonthConstants::FEBRUARY,
            'mar' => MonthConstants::MARCH,
            'apr' => MonthConstants::APRIL,
            'may' => MonthConstants::MAY,
            'jun' => MonthConstants::JUNE,
            'jul' => MonthConstants::JULY,
            'aug' => MonthConstants::AUGUST,
            'sep' => MonthConstants::SEPTEMBER,
            'oct' => MonthConstants::OCTOBER,
            'nov' => MonthConstants::NOVEMBER,
            'dec' => MonthConstants::DECEMBER,
        ];

        foreach ($months as $external => $internal) {
            if (filter_var((string) $this->object->tour->season[$external], FILTER_VALIDATE_BOOLEAN)) {
                $items[] = new AttributeAdapter($this->getExternalId(), 'bestSeason', $internal);
            }
        }

        $cleaner = new StringCleaner();
        $simpleAttributes = [
            TourAttributes::ASCEND_ELEVATION => (int) $this->object->tour->elevation['ascent'],
            TourAttributes::DESCENT_ELEVATION => (int) $this->object->tour->elevation['descent'],
            TourAttributes::CONDITION_RATING => (int) $this->object->tour->rating['condition'],
            TourAttributes::DIFFICULTY_RATING => (int) $this->object->tour->rating['difficulty'],
            TourAttributes::NECESARRY_EQUIPMENT =>
                $cleaner->cleanHtmlString((string) $this->object->tour->equipment),
            TourAttributes::EXPERIENCE_RATING => (int) $this->object->tour->rating['qualityOfExperience'],
            TourAttributes::LANDSCAPE_RATING => (int) $this->object->tour->rating['landscape'],
            TourAttributes::MAXIMUM_ALTITUDE => (int) $this->object->tour->elevation['maxAltitude'],
            TourAttributes::MINIMUM_ALTITUDE => (int) $this->object->tour->elevation['minAltitude'],
            TourAttributes::PUBLIC_TRANSIT =>
                $cleaner->cleanHtmlString((string) $this->object->tour->publicTransit),
            TourAttributes::PRIVATE_TRANSIT =>
                $cleaner->cleanHtmlString((string) $this->object->tour->gettingThere),
            TourAttributes::STARTING_POINT =>
                $cleaner->cleanHtmlString((string) $this->object->tour->startingPointDescr),
            TourAttributes::ENDING_POINT =>
                $cleaner->cleanHtmlString((string) $this->object->tour->destination),
            TourAttributes::TECHNIQUE_RATING => (int) $this->object->tour->rating['technique'],
            TourAttributes::TOUR_DURATION => (int) $this->object->tour->time['min'],
            TourAttributes::TOUR_LENGTH => (int) $this->object->tour->length,
            TourAttributes::DATA_SOURCE => TourAttributes::DATA_SOURCE_OUTDOOR_ACTIVE,
            TourAttributes::DATA_SOURCE_ID =>
                $cleaner->cleanHtmlString((string) $this->object->tour['id']),
            TourAttributes::DIRECTIONS =>
                $cleaner->cleanHtmlString((string) $this->object->tour->directions),
            TourAttributes::PARKING => $cleaner->cleanHtmlString((string) $this->object->tour->parking),
            TourAttributes::ADDITIONAL_INFORMATION =>
                $cleaner->cleanHtmlString((string) $this->object->tour->additionalInformation),
            TourAttributes::HINT_OF_AUTHOR =>
                $cleaner->cleanHtmlString((string) $this->object->tour->tip),
            TourAttributes::SAFETY_NOTE =>
                $cleaner->cleanHtmlString((string) $this->object->tour->safetyGuidelines),
        ];

        foreach ($simpleAttributes as $name => $data) {
            if (!is_string($data) || $data !== '') {
                $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
            }
        }

        if ($this->object->tour['opened'] !== null && (string) $this->object->tour['opened'] === 'false') {
            $items[] = new AttributeAdapter($this->getExternalId(), 'labels', TourAttributes::ISCLOSED);
        }

        return $items;
    }


    private function extractGeometry()
    {
        if ($this->cachedGeometry) {
            return $this->cachedGeometry;
        }

        $geometry = [];
        if ($this->object->tour->geometry) {
            $geoItems = explode(' ', (string) $this->object->tour->geometry);
            foreach ($geoItems as $geoItem) {
                $geo = explode(',', $geoItem);
                $geometry[] = [ (float) $geo[1], (float) $geo[0] ];
            }
        }

        $this->cachedGeometry = $geometry;
        return $geometry;
    }

    public function getSourceName(): ?string
    {
        $sourceName = $this->object->tour->meta->source['name'] ?? null;
        return $sourceName ? (string) $sourceName : null;
    }

    public function getAuthorName(): ?string
    {
        return null;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return null;
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }
}
