<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Promise\RejectionException;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\CongressLocationAdapter;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\OfferAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\PackageAdapter;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\RetryPool;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Psr\Log\LoggerAwareTrait;
use function Safe\json_decode;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    use LanguageAware, Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://tportal.toubiz.de';

    /**
     * Template used for detail URLs.
     * This template may contain the placeholder `{tportalUriSegment}` which will be replaced
     * with the uri segment provided by tportal.
     *
     * @var string|null
     */
    private $detailUriTemplate;

    public function fetchLodgings(callable $block): ServiceResult
    {
        $concurrency = $this->parameters['concurrency'] ?? 10;

        $pool = new ConcurrentPaginatedRequests(
            $concurrency,
            function (int $page) use ($block) {
            // By default a page has 10 results.
                if (!$this->withinLimit($page * 10)) {
                    return new RejectedPromise('Page is outside of limit');
                }

                return $this->lodgingRequest($page)->then(
                    function ($data) use ($block) {
                        foreach ($data['housedata'] as $id => $item) {
                            $block($this->lodgingDataToAdapter($id, $data), $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->start()->wait();

        // TODO fetch deleted records.
        return new ServiceResult();
    }

    public function fetchCongressLocations(callable $block): ServiceResult
    {
        $pool = new ConcurrentPaginatedRequests(
            10,
            function (int $page) use ($block) {
                return $this->congressLocationRequest($page)->then(
                    function ($data) use ($block) {
                        if (!$data) {
                            throw new RejectionException('No more data');
                        }

                        foreach ($data['housedata'] as $id => $item) {
                            $adapter = $this->congressLocationDataToAdapter($id, $data);
                            $block($adapter, $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->start()->wait();

        // TODO fetch deleted records.
        return new ServiceResult();
    }

    public function fetchOffers(callable $block): ServiceResult
    {
        $pool = new ConcurrentPaginatedRequests(
            10,
            function (int $page) use ($block) {
            // By default a page has 10 results.
                if (!$this->withinLimit($page * 10)) {
                    return new RejectedPromise('Page is outside of limit');
                }

                return $this->offerRequest($page)->then(
                    function ($data) use ($block) {
                        if (!$data) {
                            throw new RejectionException('No more data');
                        }

                    // If there is only one result then `offerData` is not an array of
                    // data objects but the data object itself.
                        if (ArrayUtility::isAssociative($data['offerData'])) {
                            $data['offerData'] = [ $data['offerData'] ];
                        }

                        foreach ($data['offerData'] as $offer) {
                            $block($this->offerDataToAdapter($offer, $data), $data['paging_txt']['count'] ?? null);
                        }
                    }
                );
            }
        );

        $pool->start()->wait();

        // TODO fetch deleted records.
        return new ServiceResult();
    }

    public function fetchPackages(callable $block)
    {
        $pool = new ConcurrentPaginatedRequests(
            5,
            function (int $page) use ($block) {
                return $this->packageRequest($page)->then(
                    function ($data) use ($block) {
                        foreach ($data['PackageAccommodationInfos'] as $package) {
                            $block($this->packageDataToAdapter($package, $data));
                        }
                    }
                );
            }
        );

        $pool->promise()->wait();

        // TODO fetch deleted records.
        return new ServiceResult();
    }

    private function lodgingDataToAdapter($id, array $data): LodgingAdapter
    {
        $lodging = new LodgingAdapter(
            [
                'housedata' => $data['housedata'][$id],
                'searchresult' => $data['searchresult'][$id][0],
                'searchresult_details' => $data['searchresult_details'][$id],
            ]
        );
        if ($this->language) {
            $lodging->setLanguage($this->language);
        }
        $lodging->setDetailUriTemplate($this->detailUriTemplate);
        return $lodging;
    }

    private function congressLocationDataToAdapter($id, array $data): CongressLocationAdapter
    {
        $congressLocation = new CongressLocationAdapter(
            [
                'housedata' => $data['housedata'][$id],
                'searchresult' => $data['searchresult'][$id][0],
                'searchresult_details' => $data['searchresult_details'][$id],
            ]
        );
        if ($this->language) {
            $congressLocation->setLanguage($this->language);
        }
        $congressLocation->setDetailUriTemplate($this->detailUriTemplate);
        return $congressLocation;
    }

    private function offerDataToAdapter(array $offer, array $data): OfferAdapter
    {
        $id = $offer['serviceID'];
        $item = new OfferAdapter(
            [
                'offer' => $offer,
                'serviceData' => $data['serviceData'][$id],
                'availability' => $data['ServiceAvailabilities'][$id] ?? [],
                'geoResult' => $this->extractGeoResult($data['geoResultData'] ?? [], $id),
            ]
        );
        if ($this->language) {
            $item->setLanguage($this->language);
        }
        return $item;
    }

    private function packageDataToAdapter(array $package, array $data): PackageAdapter
    {
        $id = $package['packageID'];
        $item = new PackageAdapter(
            [
                'package' => $package,
                'serviceData' => $data['serviceData'][$id],
            ]
        );
        if ($this->language) {
            $item->setLanguage($this->language);
        }
        return $item;
    }

    /**
     * @param array $geoResultData
     * @param string $id
     * @return array|null
     */
    private function extractGeoResult(array $geoResultData, string $id)
    {
        foreach ($geoResultData as $data) {
            if (($data['serviceID'] ?? null) === $id) {
                return $data;
            }
        }
        return null;
    }

    public function setDetailUriTemplate(?string $detailUriTemplate): void
    {
        $this->detailUriTemplate = $detailUriTemplate;
    }

    private function offerRequest(int $page, int $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/offer?reset=1&json=1&page=:page',
            [
                ':clientName' => $this->clientName,
                ':page' => $page,
            ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                return is_array($data) && array_key_exists('offerData', $data) && \count($data['offerData']) > 0;
            }
        );
    }

    private function packageRequest(int $page, int $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/package?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                return array_key_exists('PackageAccommodationInfos', $data)
                    && \count($data['PackageAccommodationInfos']) > 0;
            }
        );
    }

    private function lodgingRequest(int $page, $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/ukv/search?reset=1&json=1&ukv_result_order=3&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                return $data && $data['housedata'] && count($data['housedata']) > 0;
            }
        );
    }

    private function congressLocationRequest(int $page, $retriesIfFail = 5): PromiseInterface
    {
        $url = $this->urlTemplate(
            '/:clientName/congress?reset=1&ukv_result_order=3&json=1&page=:page',
            [
                ':clientName' => $this->clientName,
                ':page' => $page,
            ]
        );
        $url = $this->addLanguageToUrl($url);

        return $this->jsonRequest(
            new Uri($url),
            $retriesIfFail,
            function ($data) {
                return array_key_exists('housedata', $data) && count($data['housedata']) > 0;
            }
        );
    }

    private function urlTemplate(string $url, array $replacements): string
    {
        return str_replace(
            array_keys($replacements),
            array_map('urlencode', array_values($replacements)),
            $url
        );
    }

    private function addLanguageToUrl(string $url): string
    {
        if (!$this->language) {
            return $url;
        }

        return $url . '&lang=' . $this->language;
    }
}
