<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;

/**
 * Point of interest adapter.
 *
 * This represents an Article from the DB service.
 */
class PointOfInterestAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    ArticleWithSocialMediaLinksAdapterInterface
{
    use LanguageAware;
    use WithCacheProperty;
    use UsesFirstMediumAsMainMedium;

    /**
     * @var CategoryMap
     */
    private $categoryMap;

    public function __construct($adaptee, CategoryMap $categoryMap)
    {
        parent::__construct($adaptee);
        $this->categoryMap = $categoryMap;
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_ATTRACTION;
    }

    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    public function getName(): string
    {
        return substr($this->object['name'], 0, 240);
    }

    public function getAbstract(): ?string
    {
        $value = $this->object['object']['data_map']['intro']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    public function getDescription(): ?string
    {
        $value = $this->object['object']['data_map']['description']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    public function getLatitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[0];
        }

        return null;
    }

    public function getLongitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[1];
        }

        return null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $categories = [];

        // We omit the main category from 'main_type' because it is always included in the categories from 'type'.
        $categoryData = $this->object['object']['data_map']['type'];
        $categoryIds = explode('#', $categoryData['remote_ids']);

        foreach ($categoryIds as $categoryId) {
            $name = $this->categoryMap->getName($categoryId);
            // New categories might not be in the static category map, yet.
            // We just skip these.
            /** @todo add notification */
            if ($name !== '') {
                $categories[] = new CategoryAdapter(
                    [
                        'remote_id' => $categoryId,
                        'name' => $name,
                    ]
                );
            }
        }

        $mainCategoryId = $this->object['object']['data_map']['type_main']['remote_id'];
        $categories = $this->moveMainCategoryToTop($categories, $mainCategoryId);

        return $categories;
    }

    /**
     * @param array $categories
     * @param string $mainCategoryId
     * @return CategoryAdapter[]
     */
    public function moveMainCategoryToTop(array $categories, string $mainCategoryId): array
    {
        usort(
            $categories,
            function (CategoryAdapter $categoryA, CategoryAdapter $categoryB) use ($mainCategoryId) {
                return $categoryA->getExternalId() === $mainCategoryId ? -1 : 1;
            }
        );

        return $categories;
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache(
            'media',
            function () {
                $media = [];
                if (!is_array($this->object['object']['data_map']['gallery']['content'])) {
                    return $media;
                }

                foreach ($this->object['object']['data_map']['gallery']['content'] as $key => $image) {
                    // Skip importing media records without a source uri.
                    if (!empty($image['reference'])) {
                        // The array key defines the sorting order which must be preserved.
                        $image['_sorting'] = (int) $key;
                        $media[] = (new MediumAdapter($image));
                    }
                }
                return $media;
            }
        );
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return AttributeAdapter[]
     */
    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    public function getFiles(): array
    {
        // TODO
        return [];
    }

    private function getSocialUri(string $datamapItemName): ?string
    {
        $value = $this->object['object']['data_map'][$datamapItemName]['content'] ?? null;
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return null;
        }

        return (string) $value;
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialUri('flickr');
    }

    public function getSourceName(): ?string
    {
        return null;
    }

    public function getAuthorName(): ?string
    {
        return $this->object['object']['author'] ?? null;
    }

    protected function isUrlPlaceholder(string $value): bool
    {
        return ($value === 'http://' || $value === 'https://');
    }

    public function getOpeningTimes(): ?string
    {
        if (!array_key_exists('opentimes', $this->object)) {
            return null;
        }

        $openTimes = json_encode($this->object['opentimes']);
        if ($openTimes === false) {
            return null;
        }
        return $openTimes;
    }

    public function getOpeningTimesFormat(): ?string
    {
        return $this->getOpeningTimes() === null ? null : ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    public function getLevelOfMaintenance(): int
    {
        return (int) $this->object['object']['data_map']['level']['content'];
    }

    private function parseAttributes(): array
    {
        $attributes = [];

        $featureList = $this->getFeatureList();
        if (!empty($featureList)) {
            $attributes[PointOfInterestAttributes::FEATURES] = $featureList;
        }

        if (!empty($this->object['object']['data_map']['tags']['content'])) {
            $attributes[PointOfInterestAttributes::TAG] = ArrayUtility::trimExplode(
                ',',
                $this->object['object']['data_map']['tags']['content']
            );
        }

        foreach ($this->dataMapMapping as $normalized => $external) {
            if (!empty($this->object['object']['data_map'][$external]['content'])) {
                $attributes[$normalized] = $this->object['object']['data_map'][$external]['content'];
            }
        }

        $childrenList = $this->getChildrenList();
        foreach ($this->extensionAttributeMapping as $extension => $mapping) {
            foreach ($mapping as $normalized => $external) {
                if (!empty($childrenList[$extension]['object']['data_map'][$external]['content'])) {
                    $attributes[$normalized] = $childrenList[$extension]['object']['data_map'][$external]['content'];
                }
            }
        }

        return $attributes;
    }

    private function getChildrenList(): array
    {
        $children = [];
        foreach ($this->object['object_children_list'] ?? [] as $child) {
            foreach ($child as $name => $content) {
                $children[$name] = $content;
            }
        }
        return $children;
    }

    private function getFeatureList(): array
    {
        $features = [];

        // Group names are not translated via API, so we add generic translations.
        $map = [
            'd9fb978199b5a88bc6205ed79e75c9f0' => 'information',
            'e4506d15ec7dd2200fb01c6e0484493e' => 'acceptedPayments',
            '4ad54417b84ec9ee6491b006f1448ff9' => 'forKids',
            '6233065458a995d42a70ef99e6fdc434' => 'spokenLanguages',
            '555eaacf11d1037df2d840df2ecee09b' => 'priceSegment',
        ];

        foreach ($this->object['object']['data_map']['feature'] ?? [] as $feature) {
            $parent = $feature['parent_id'] ?? null;
            $group = $map[$parent] ?? null;
            $content = $feature['content'] ?? null;

            if ($group === null || $content === null) {
                continue;
            }

            if (!\array_key_exists($group, $features)) {
                $features[$group] = [];
            }

            if (!empty($feature)) {
                $features[$group][] = $content;
            }
        }

        return array_filter($features);
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }


    private $dataMapMapping = [
        PointOfInterestAttributes::FREE_ENTRANCE => 'prices_free',
        PointOfInterestAttributes::PRICE_INFORMATION => 'priceinfotext',
        PointOfInterestAttributes::PRICES => 'prices',
        PointOfInterestAttributes::DISCOUNT_CARD => 'discount_card',
    ];

    private $extensionAttributeMapping = [
        'therme_extension' => [
            PointOfInterestAttributes::THERMAL_CLASSIFICATION => 'klassifizierung',
            PointOfInterestAttributes::THERMAL_POOLQUANTITYINSSIDE => 'anzahl_becken_innen',
            PointOfInterestAttributes::THERMAL_POOLAREAINSSIDE => 'flaeche_becken_innen',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATURESTARTINSIDE => 'temperatur_becken_innen_von',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATUREENDINSIDE => 'temperatur_becken_innen_bis',
            PointOfInterestAttributes::THERMAL_POOLQUANTITYOUTSIDE => 'anzahl_becken_aussen',
            PointOfInterestAttributes::THERMAL_POOLAREAOUTSIDE => 'flaeche_becken_aussen ',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATURESTARTOUTSIDE => 'temperatur_becken_aussen_von',
            PointOfInterestAttributes::THERMAL_POOLTEMPERATUREENDOUTSIDE => 'temperatur_becken_aussen_bis',
            PointOfInterestAttributes::THERMAL_POOLFEATURES => 'ausstattung_badebereich',
            PointOfInterestAttributes::THERMAL_POOLFEATURESADDITIONAL => 'zusaetzliche_ausstattung_therme',
            PointOfInterestAttributes::THERMAL_WATERTYPE => 'wasser',
            PointOfInterestAttributes::THERMAL_WATERCLEANING => 'reinigung_wasser',
            PointOfInterestAttributes::THERMAL_WATERDEPTHAVERAGE => 'durchschnittliche_wassertiefe',
            PointOfInterestAttributes::THERMAL_HASNONSWIMMMERSEPERATION => 'abtrennung_nichtschwimmer',
            PointOfInterestAttributes::THERMAL_HASTHERAPYPOOL => 'therapiebecken',
            PointOfInterestAttributes::THERMAL_HASSPORTPOOL => 'sport_becken',
            PointOfInterestAttributes::THERMAL_HASUNDERWATERMUSIC => 'becken_unterwassermusik',
            PointOfInterestAttributes::THERMAL_HASNATUREBATHINGLAKE => 'naturbadesee',
            PointOfInterestAttributes::THERMAL_HASWHIRLPOOL => 'whirlpool',
            PointOfInterestAttributes::THERMAL_HASPOOLMASSAGEJET => 'becken_massageduesen',
            PointOfInterestAttributes::THERMAL_HASFLUXPOOL => 'stroemungsbecken',
            PointOfInterestAttributes::THERMAL_HASSEAPOOL => 'meeresbecken',
            PointOfInterestAttributes::THERMAL_HASSINGLECABIN => 'einzelkabinen',
            PointOfInterestAttributes::THERMAL_HASFAMILYCABIN => 'familienkabinen',
            PointOfInterestAttributes::THERMAL_HASWARDROBE => 'kleiderschraenke',
            PointOfInterestAttributes::THERMAL_HASLOCKER => 'schliessfaecher',
            PointOfInterestAttributes::THERMAL_HASHAIRDRYER => 'haartrockner',
            PointOfInterestAttributes::THERMAL_HASSHOWERS => 'duschen',
            PointOfInterestAttributes::THERMAL_HASEMERGENCYROOM => 'sanitaetsraum',
            PointOfInterestAttributes::THERMAL_HASTOWELRENTAL => 'handtuch_verleih',
            PointOfInterestAttributes::THERMAL_HASBATHROBERENTAL => 'bademantel_verleih',
            PointOfInterestAttributes::THERMAL_HASSHOP => 'shop',
            PointOfInterestAttributes::THERMAL_HASGASTRONOMY => 'gastronomie',
            PointOfInterestAttributes::THERMAL_SAUNA_FEATURES => 'saunaausstattung',
            PointOfInterestAttributes::THERMAL_SAUNA_ADDITIONALFEATURES => 'ausstattung_saunabereich',
            PointOfInterestAttributes::THERMAL_SAUNA_QUANTITY => 'anzahl_saunen',
            PointOfInterestAttributes::THERMAL_SAUNA_MINIMIUMTEMPERATURE => 'mindesttemperatur',
            PointOfInterestAttributes::THERMAL_SAUNA_MAXIMUMTEMPERATURE => 'maximaltemperatur',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONQUARTERHOURLY => 'viertelstuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONHALFHOURLY => 'halbstuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_ISINFUSIONHOURLY => 'stuendlich',
            PointOfInterestAttributes::THERMAL_SAUNA_INFUSIONINTERVAL => 'zeitliche_taktung',
            PointOfInterestAttributes::THERMAL_SAUNA_COLDAPLICATION => 'abkuehl_angebote',
            PointOfInterestAttributes::THERMAL_SAUNA_QUIETROOM => 'ruheraeume',
            PointOfInterestAttributes::THERMAL_SAUNA_QUIETROOMQUANTITY => 'anzahl_ruheraeme',
            PointOfInterestAttributes::THERMAL_SAUNA_SERVICE => 'service_sauna',
            PointOfInterestAttributes::THERMAL_WELLNESS_APPLICATIONS => 'wellness_anwendungen',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASMASSAGES => 'wellnessmassagen',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASEXOTIC => 'exotische_wellnessangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASSOFTPACK => 'softpack',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASTUBBATH => 'wannenbaeder',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASPEELING => 'koerperpeeling',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASCOSMETIC => 'kosmetikangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASRELAXATION => 'entspannungsangebote',
            PointOfInterestAttributes::THERMAL_WELLNESS_HASPRIVATESPA => 'private_spa',
            PointOfInterestAttributes::THERMAL_FITNESS_INDOOR => 'fitness_indoor',
            PointOfInterestAttributes::THERMAL_FITNESS_OUTDOOR => 'fitness_outdoor',
            PointOfInterestAttributes::THERMAL_HASHOTEL => 'thermalHotel',
            PointOfInterestAttributes::THERMAL_CAMPERQUANTITY => 'anzahl_reisemobil',
            PointOfInterestAttributes::THERMAL_FAMILYCHILDAGEMINIMUM => 'mindestalter_kinder',
            PointOfInterestAttributes::THERMAL_ISFAMILYFRIENDLY => 'familiengeeignet',
            PointOfInterestAttributes::THERMAL_ISWHEELCHAIRACCESSIBLE => 'rollstuhlgerecht',
            PointOfInterestAttributes::THERMAL_ISACCESSIBLE => 'behindertengerecht',
        ],
        'barrierfree_new' => [
            PointOfInterestAttributes::ACCESSIBLE_IS90 => 'barrierfree_90',
            PointOfInterestAttributes::ACCESSIBLE_IS80 => 'barrierfree_80',
            PointOfInterestAttributes::ACCESSIBLE_IS70 => 'barrierfree_70',
            PointOfInterestAttributes::ACCESSIBLE_IS60 => 'barrierfree_60',
            PointOfInterestAttributes::ACCESSIBLE_ISFORDEAFS => 'barrierfree_ear',
            PointOfInterestAttributes::ACCESSIBLE_ISFORBLINDS => 'barrierfree_eye',
            PointOfInterestAttributes::ACCESSIBLE_ISFORDISORDERS => 'barrierfree_lern',
            PointOfInterestAttributes::ACCESSIBLE_TEXT => 'barrierfree_text',
            PointOfInterestAttributes::ACCESSIBLE_HASELEVATOR => 'barrierfree_lift',
            PointOfInterestAttributes::ACCESSIBLE_ELEVATORWIDTH => 'barrierfree_lift_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWC => 'barrierfree_wc',
            PointOfInterestAttributes::ACCESSIBLE_WCWIDTH => 'barrierfree_wc_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWCPARTWIDTH => 'barrierfree_wc_part_width',
            PointOfInterestAttributes::ACCESSIBLE_HASWCPART => 'barrierfree_wc_part',
        ],
    ];
}
