<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Promise\RejectedPromise;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Guzzle\ConcurrentPaginatedRequests;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMapParser;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\CityApiService\CityAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;

/**
 * Service for legacy Toubiz DB API.
 */
class CityApiService extends AbstractService
{
    use LanguageAware, Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://db-service.toubiz.de/';

    /**
     * Map of language constants to URI segments expected by the API.
     *
     * @var array
     */
    private static $languageMap = [
        Language::DE => 'ger-DE',
        Language::EN => 'eng-GB',
        Language::FR => 'fre-FR',
        Language::ES => 'esl-ES',
    ];

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Article' => PointOfInterestAdapter::class,
    ];
    const CATEGORY_MAP_CSV = '/../../../../Resources/Service/Toubiz/Legacy/DbService/categories-poi.csv';

    public function fetchCities(callable $block): ServiceResult
    {
        $concurrency = $this->parameters['concurrency'] ?? 10;
        $perPage = $this->parameters['perPage'] ?? 25;

        $pool = new ConcurrentPaginatedRequests($concurrency, function (int $page) use ($perPage, $block) {
            if (!$this->withinLimit($page * $perPage)) {
                return new RejectedPromise('Page is outside of limit');
            }
            return $this->sendRequest(($page - 1) * $perPage, $perPage)->then(function (array $data) use ($block) {
                foreach ($data['children_list'] as $item) {
                    $block($this->cityDataToAdapter($item), $data['children_count'] ?? null);
                }
            });
        });

        $pool->start()->wait();

        // TODO figure out how to fetch deleted cities
        return new ServiceResult();
    }


    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $offset
     * @param int $limit
     * @param int $retriesIfFail
     * @return PromiseInterface
     */
    protected function sendRequest($offset, $limit, $retriesIfFail = 5): PromiseInterface
    {
        $parameters = [
            ':apiKey' => $this->apiKey,
            ':clientName' => $this->clientName,
            ':offset' => $offset,
            ':limit' => $limit,
            ':modified' => $this->delta->asSeconds(),
        ];

        $url = str_replace(
            array_keys($parameters),
            $parameters,
            '/layout/set/json/content/view/json/:clientName/' .
            '(key)/:apiKey/(class)/city/(modified)/:modified/(offset)/:offset/(limit)/:limit'
        );
        $url = $this->appendLanguageToUrl($url);

        return $this->jsonRequest(new Uri($url), $retriesIfFail, function (array $data) {
            return $data['content'] && count($data['content']['children_list']) > 0;
        })->then(function (array $data) {
            return $data['content'];
        });
    }

    private function cityDataToAdapter(array $item): CityAdapter
    {
        $article = new CityAdapter($item);
        if ($this->language) {
            $article->setLanguage($this->language);
        }
        return $article;
    }

    /**
     * The category map is built from a static csv export from Toubiz Legacy.
     * This should be updated when new categories are added in Toubiz Legacy Backend.
     */
    private function getCategoryMap(): CategoryMap
    {
        $file = __DIR__ . self::CATEGORY_MAP_CSV;

        return (new CategoryMapParser())->parse($file, (string) $this->language);
    }

    private function appendLanguageToUrl(string $url): string
    {
        $languageUrlSegment = static::$languageMap[$this->language] ?? null;
        if ($languageUrlSegment) {
            return sprintf('%s/(lang)/%s', $url, $languageUrlSegment);
        }
        return $url;
    }
}
