<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\DirectMarketerAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\AttributeAdapter;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;

/**
 * Direct marketer adapter.
 *
 * This represents an Article with mapping for the Toubiz specific direct marketer.
 */
class DirectMarketerAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_DIRECT_MARKETER;
    }

    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getDescription(): ?string
    {
        $description = null;
        $value = '';
        if (!is_array($this->object['description'])) {
            return $description;
        }
        if (array_key_exists('direkt_event', $this->object['description'])) {
            $value .= $this->object['description']['direkt_event']['text'];
        }
        if (array_key_exists('direkt', $this->object['description'])) {
            $value .= $this->object['description']['direkt']['text'];
        }


        if (!empty($value)) {
            $description = $value;
        }

        return $description;
    }

    public function getLatitude(): ?float
    {
        return $this->object['location']['Latitude'] ?? null;
    }

    public function getLongitude(): ?float
    {
        return $this->object['location']['Longitude'] ?? null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        if (!\is_array($this->object['images'])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $media[] = (new MediumAdapter($image));
        }
        return $media;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        return AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialMediaUri('facebook');
    }


    public function getTwitterUri(): ?string
    {
        return $this->getSocialMediaUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialMediaUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialMediaUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialMediaUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialMediaUri('flickr');
    }

    private function getSocialMediaUri(string $socialMedia): ?string
    {
        if (!array_key_exists('social_media', $this->object)
            ||!is_array($this->object['social_media'])
        ) {
            return null;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            $socialMedia
        );

        return $data ? $data['value'] : null;
    }

    public function getSourceName(): ?string
    {
        return $this->object['headline_quellmandant'] ?? null;
    }

    public function getAuthorName(): ?string
    {
        $author = null;
        if (!empty($this->object['author'])) {
            $author = $this->object['author'];
        }

        return $author;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] ?? null) === '1';
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $keys = [
            'dv_marketing' => DirectMarketerAttributes::SHOPPING,
            'dv_prod_cat_aufstriche' => DirectMarketerAttributes::OIL,
            'dv_prod_cat_gemuese_obst' => DirectMarketerAttributes::VEGETABLES,
            'dv_prod_cat_getraenke' => DirectMarketerAttributes::DRINKS,
            'dv_prod_cat_saison' => DirectMarketerAttributes::SEASONALS,
            'dv_special' => DirectMarketerAttributes::COMMODITIES,
            'dv_prod_cat_fleisch' => DirectMarketerAttributes::MEATS,
            'dv_prod_cat_gefluegel_fisch' => DirectMarketerAttributes::FISH,
            'dv_prod_cat_milch' => DirectMarketerAttributes::MILK,
            'dv_prod_cat_getreide' => DirectMarketerAttributes::CEREALS,
            'dv_prod_cat_handwerk' => DirectMarketerAttributes::CRAFTS,
            'dv_decoration' => DirectMarketerAttributes::AWARDS,
            'dv_decoration_regional' => DirectMarketerAttributes::REGIONAL_AWARDS,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        return $attributes;
    }

    /**
     * Extract attributes from a volatile set of data.
     *
     * @param string $key
     * @return array|null
     */
    private function extractAttributes($key)
    {
        $items = [];

        // The desired key can be in two different places.
        if (array_key_exists($key, $this->object)) {
            $attributes = $this->object[$key];
        } elseif (is_array($this->object['property'])
            && array_key_exists($key, $this->object['property'])
        ) {
            $attributes = $this->object['property'][$key];
        } elseif (is_array($this->object['classification'])
            && array_key_exists($key, $this->object['classification'])
        ) {
            $attributes = $this->object['classification'][$key];
        } else {
            return null;
        }

        // Also, the structure is different for finding the attributes.
        if (is_array($attributes[0])
            && array_key_exists('properties', $attributes[0])
        ) {
            $attributes = $attributes[0]['properties'];
        }

        foreach ($attributes as $entry) {
            // For the direct Marketers Value is never used with exception for the awards
            $items[] = $entry['text'];
        }

        if (!empty($items)) {
            return $items;
        }
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getFiles(): array
    {
        return [];
    }
}
