<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService\DirectMarketerAdapter;

/**
 * Service for legacy Toubiz gastronomy API.
 *
 * Concrete implementation to communicate with
 * the first, "old" toubiz API returning gastronomy articles.
 */
class DirectMarketerApiService extends AbstractService
{
    use LanguageAware;
    use Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://api.toubiz.de/json/direkt/';

    /**
     * Fetch records.
     *
     * @param callable $block
     * @return void
     */
    public function fetchDirectMarketers(callable $block): void
    {
        $offset = 0;
        $limit = 100;
        while ($this->withinLimit($offset + $limit)) {
            $data = $this->sendRequest($offset, $limit);
            if (!$data) {
                return;
            }

            foreach ($data as $item) {
                $article = new DirectMarketerAdapter($item);
                if ($this->language) {
                    $article->setLanguage($this->language);
                }
                $block($article);
            }

            $offset += $limit;
        }
    }

    /**
     * Requests data from the API.
     *
     * @param int $offset
     * @param int $limit
     * @return mixed
     */
    protected function sendRequest($offset, $limit)
    {
        $response = $this->httpClient->request(
            'GET',
            'modified.php',
            [
                'query' => [
                    'key' => $this->apiKey,
                    'modified' => 86400000000,
                    'offset' => $offset,
                    'limit' => $limit
                ],
            ]
        );
        if ($response->getStatusCode() === 200) {
            $dataRaw = $response->getBody();
            $dataRaw = StringCleaner::asString($dataRaw);
            $data = json_decode($dataRaw, true);

            if ($data === null) {
                throw new InvalidJsonException(
                    sprintf('API request returned invalid JSON (offset %d, limit %d).', $offset, $limit),
                    1557149751
                );
            }

            if ($data['result']) {
                return $data['result']['item'];
            }

            return false;
        }

        return false;
    }
}
