<?php
namespace Newland\Toubiz\Api\Service\Meteotest\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\Weather\DayAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\WeatherAdapterInterface;

class WeatherAdapter implements WeatherAdapterInterface
{
    public static $dictionary = [
        'dd' => 'windDirection',
        'ff' => 'windSpeed',
        'fx' => 'maximumWindGust',
        'rr' => 'rainPrecipitation',
        'rrp' => 'rainProbability',
        'sl' => 'snowfallLine',
        'ss' => 'sunshineDuration',
        'sy' => 'symbol',
        'tn' => 'minimumTemperature',
        'tt' => 'temperature',
        'tx' => 'maximumTemperature',
    ];
    /**
     * @var DayAdapter[]
     */
    protected $days;
    /**
     * @var string
     */
    private $location;
    /**
     * @var int
     */
    private $lastUpdated;

    const IDENTIFIER_DATE_FORMAT = 'Ymd';

    public function __construct(string $location, array $data)
    {
        $this->location = $location;
        $this->parseData($data);
        $this->lastUpdated = $this->getCurrentTime();
    }

    private function parseData(array $data)
    {
        $dateToPopulate = $this->getDateShortFormat(array_keys($data)[0]);
        $hourRecordsOfCurrentDay = [];

        $lastItemDateString = $this->getLastItemDateString($data);
        $dateComplete = false;

        foreach ($data as $dateString => $values) {
            $currentItemDate = $this->getDateShortFormat($dateString);

            if ($dateToPopulate === $currentItemDate) {
                $hourRecordsOfCurrentDay[$dateString] = $values;
            }

            $dateComplete = $dateToPopulate !== $currentItemDate || $dateString === $lastItemDateString;

            if ($dateComplete) {
                $this->days[] = new DayAdapter($dateToPopulate, $hourRecordsOfCurrentDay);
                $dateToPopulate = $currentItemDate;
                $hourRecordsOfCurrentDay = [
                    $dateString => $values,
                ];
                $dateComplete = false;
            }
        }
    }

    private function getLastItemDateString(array $data): string
    {
        $dateStrings = array_keys($data);
        return array_pop($dateStrings);
    }

    private function getDateShortFormat(string $dateString): string
    {
        $lastDate = new \DateTime($dateString);
        return $lastDate->format(self::IDENTIFIER_DATE_FORMAT);
    }

    /**
     * Wrap in public function for mocking.
     *
     * @return int
     */
    public function getCurrentTime(): int
    {
        return time();
    }

    public function getExternalId(): string
    {
        return $this->getLocation();
    }

    public function getLocation(): string
    {
        return $this->location;
    }

    /**
     * @return DayAdapterInterface[]
     */
    public function getForecast(): array
    {
        return $this->days;
    }

    /**
     * @return int Timestamp of latest import
     */
    public function getLastUpdated(): int
    {
        return $this->lastUpdated;
    }
}
