<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMapParser;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService\PointOfInterestAdapter;
use Newland\Toubiz\Api\Utility\RetryPool;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for legacy Toubiz DB API.
 */
class DbService extends AbstractService
{
    use LanguageAware;
    use Limitable;
    use LoggerAwareTrait;
    use SharedLegacyToubizServiceBehaviour;

    const DEFAULT_BASE_URI = 'https://db-service.toubiz.de/';

    public function fetchPointOfInterests(callable $block): void
    {
        $offset = 0;
        $limit = 25;
        $categoryMap = $this->getCategoryMap();
        while ($this->withinLimit($offset + $limit)) {
            $data = $this->sendRequest($offset, $limit);
            if (!$data) {
                return;
            }

            foreach ($data as $item) {
                $article = new PointOfInterestAdapter($item, $categoryMap);
                $article->setLogger($this->logger);
                if ($this->language) {
                    $article->setLanguage($this->language);
                }
                $block($article);
            }

            $offset += $limit;
        }
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $offset
     * @param int $limit
     * @param int $retriesIfFail
     * @return mixed
     */
    protected function sendRequest($offset, $limit, $retriesIfFail = 5)
    {
        $url = $this->buildUrl([
            'class' => 'attraction',
            'modified' => '864000000',
            'offset' => $offset,
            'limit' => $limit
        ]);

        $pool = new RetryPool($retriesIfFail);
        $pool->setLogger($this->logger);
        $response = $pool->retryOnException(function () use ($url) {
            return $this->httpClient->request('GET', $url);
        });
        if ($response->getStatusCode() === 200) {
            $dataRaw = $response->getBody();
            $dataRaw = StringCleaner::asString($dataRaw);
            $data = json_decode($dataRaw, true);

            if ($data === null) {
                throw new InvalidJsonException(
                    sprintf('API request returned invalid JSON (offset %d, limit %d).', $offset, $limit),
                    1557149751
                );
            }

            if ($data['content'] && count($data['content']['children_list']) > 0) {
                return $data['content']['children_list'];
            }
        }

        return false;
    }
}
