<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use GuzzleHttp\Client;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Response;
use Newland\Toubiz\Api\Utility\RetryPool;
use Psr\Http\Message\UriInterface;
use function Safe\json_decode;

/**
 * Abstract service class.
 */
abstract class AbstractService
{
    /**
     * Abstract constant, must be set in concrete classes.
     *
     * @var string Base URI for requests.
     */
    const DEFAULT_BASE_URI = '';

    /**
     * @var string The API key matching the client name.
     */
    protected $apiKey = '';

    /**
     * @var \GuzzleHttp\Client Client for requests.
     */
    protected $httpClient;

    /**
     * @var string The client name to fetch data for.
     */
    protected $clientName = '';

    /**
     * @var array various service-specific parameters.
     */
    protected $parameters = [];

    /**
     * @api
     * @param string|null $baseUri
     */
    public function __construct(string $baseUri = null)
    {
        if (empty(static::DEFAULT_BASE_URI)) {
            throw new \InvalidArgumentException('DEFAULT_BASE_URI must be declared but is not!');
        }

        $this->httpClient = new Client([ 'base_uri' => $baseUri ?? static::DEFAULT_BASE_URI ]);
    }

    public function setHttpClientSettings(array $config): void
    {
        $config = (array) array_replace_recursive($this->httpClient->getConfig(), $config);
        $this->httpClient = new Client($config);
    }

    /**
     * Factory method to fetch data from a service.
     *
     * @param string $thing Thing to fetch, used as method name for the effective service.
     * @param callable $block Method block that is being executed for one thing.
     * @return void
     *
     * @deprecated Use `fetch*` methods directly. Magic methods are being phased out.
     */
    public function fetch($thing, callable $block): void
    {
        $methodName = 'fetch' . ucwords($thing);
        $this->$methodName($block);
    }

    /**
     * Sets the client name.
     *
     * @api
     * @param string $clientName
     * @return void
     */
    public function setClientName($clientName): void
    {
        $this->clientName = $clientName;
    }

    /**
     * Sets the api key.
     *
     * @api
     * @param string $apiKey
     * @return void
     */
    public function setApiKey($apiKey): void
    {
        $this->apiKey = $apiKey;
    }

    public function setParameters(array $parameters): void
    {
        $this->parameters = $parameters;
    }


    protected function jsonRequest(
        UriInterface $url,
        int $retriesIfFail,
        \Closure $validateData = null
    ): PromiseInterface {
        $pool = new RetryPool($retriesIfFail);
        $pool->setLogger($this->logger);

        /** @var PromiseInterface $promise */
        $promise = $pool->retryOnException(
            function () use ($url) {
                return $this->httpClient->requestAsync('GET', (string) $url);
            }
        );

        return $promise->then(
            function (Response $response) use ($validateData) {
                return $this->jsonResponse($response, $validateData);
            }
        );
    }


    protected function jsonResponse(Response $response, \Closure $validateData = null): ?array
    {
        $validateData = $validateData ?? function () {
                return true;
            };
        if ($response->getStatusCode() !== 200) {
            return null;
        }

        $dataRaw = (string) $response->getBody();
        $dataRaw = StringCleaner::asString($dataRaw);
        $data = json_decode($dataRaw, true);

        if ($validateData($data)) {
            return $data;
        }
        return null;
    }
}
