<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    /**
     * @return int
     */
    public function getMainType()
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->object['name'];
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        return null;
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        if (!is_array($this->object['description'])) {
            return null;
        }

        $value = null;
        if (array_key_exists('gastro', $this->object['description'])) {
            $value = $this->object['description']['gastro']['text'];
        } elseif (array_key_exists('gastro_event', $this->object['description'])) {
            $value = $this->object['description']['gastro_event']['text'];
        }

        return $value;
    }

    /**
     * @return array<AddressAdapterInterface>
     */
    public function getAddresses()
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress()
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return array<CategoryAdapter>
     */
    public function getCategories()
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia()
    {
        if (!is_array($this->object['images'])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $media[] = (new MediumAdapter($image));
        }
        return $media;
    }

    public function hasAttributes()
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return array
     */
    public function getAttributes()
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    /**
     * @return string
     */
    public function getFacebookUri()
    {
        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'facebook'
        );

        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getTwitterUri()
    {
        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'twitter'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getInstagramUri()
    {
        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'instagram'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return string
     */
    public function getYoutubeUri()
    {
        if (!is_array($this->object['social_media'])) {
            return '';
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            'youtube'
        );
        return $data ? $data['value'] : '';
    }

    /**
     * @return array<FileAdapter>
     */
    public function getFiles()
    {
        if (!is_array($this->object['files'])) {
            return [];
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return $this->object['headline_quellmandant'];
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        if (!empty($this->object['author'])) {
            return $this->object['author'];
        }
        return null;
    }

    /**
     * @return string
     */
    public function getBookingUri()
    {
        if (!is_array($this->object['property'])
            || !array_key_exists('gastro_lunchgate', $this->object['property'])
            || !is_array($this->object['property']['gastro_lunchgate'])
        ) {
            return '';
        }

        $value = $this->object['property']['gastro_lunchgate'][0]['properties'][0]['value'];

        // This is wrong data due to missing valitations.
        if (stristr($value, '@')) {
            return '';
        }

        return $value;
    }

    /**
     * @return string|null
     */
    public function getDetailUri()
    {
        return null;
    }

    /**
     * @return bool
     */
    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    /**
     * Return the star classification for hotels.
     *
     * @return int
     */
    public function getStarClassification()
    {
        return 0;
    }

    /**
     * Return the average rating on a scale of 0-100
     *
     * @return int
     */
    public function getAverageRating()
    {
        return 0;
    }

    /**
     * Return the number of ratings.
     *
     * @return int
     */
    public function getNumberOfRatings(): int
    {
        return 0;
    }

    /**
     * Extract attributes from a volatile set of data.
     *
     * @param string $key
     * @return array|null
     */
    private function extractAttributes($key)
    {
        $items = [];

        // The desired key can be in two different places.
        if (array_key_exists($key, $this->object)) {
            $attributes = $this->object[$key];
        } elseif (is_array($this->object['property'])
            && array_key_exists($key, $this->object['property'])
        ) {
            $attributes = $this->object['property'][$key];
        } else {
            return null;
        }

        // Also, the structure is different for finding the attributes.
        if (is_array($attributes[0])
            && array_key_exists('properties', $attributes[0])
        ) {
            $attributes = $attributes[0]['properties'];
        }

        foreach ($attributes as $entry) {
            /*
             * Most of the time, text is only used. But if it also
             * has a value given, it needs to be concatenated.
             */
            if (!empty($entry['value'])) {
                $items[] = $entry['text'] . ': ' . $entry['value'];
            } else {
                $items[] = $entry['text'];
            }
        }

        if (!empty($items)) {
            return $items;
        }
    }

    /**
     * The complete opening times data.
     *
     * @return null
     */
    public function getOpeningTimes()
    {
        return null;
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $keys = [
            'gastro_ambient' => GastronomyAttributes::AMBIANCE,
            'gastro_child' => GastronomyAttributes::FAMILIES,
            'gastro_group' => GastronomyAttributes::GROUPS,
            'gastro_kitchen' => GastronomyAttributes::KITCHEN_STYLE,
            'gastro_style' => GastronomyAttributes::GASTRONOMY_STYLE,
            'gastro_person' => GastronomyAttributes::CAPACITY,
            'gastro_payment' => GastronomyAttributes::ACCEPTED_PAYMENTS,
            'gastro_pricesegment' => GastronomyAttributes::PRICE_SEGMENT,
            'gastro_property' => GastronomyAttributes::SERVICES,
            'gastro_language' => GastronomyAttributes::SPOKEN_LANGUAGES,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        return $attributes;
    }
}
