<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Point of interest adapter.
 *
 * This represents an Article from the DB service.
 */
class PointOfInterestAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    /**
     * @var CategoryMap
     */
    private $categoryMap;

    public function __construct($adaptee, CategoryMap $categoryMap)
    {
        parent::__construct($adaptee);
        $this->categoryMap = $categoryMap;
    }

    /**
     * @return int
     */
    public function getMainType(): int
    {
        return ArticleConstants::TYPE_ATTRACTION;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    /**
     * @return string
     */
    public function getName()
    {
        return substr($this->object['name'], 0, 240);
    }

    /**
     * @return string|null
     */
    public function getAbstract()
    {
        $value = $this->object['object']['data_map']['intro']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function getDescription()
    {
        $value = $this->object['object']['data_map']['description']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    /**
     * @return array<AddressAdapterInterface>
     */
    public function getAddresses()
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress()
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return array<CategoryAdapter>
     */
    public function getCategories()
    {
        $categories = [];

        // We omit the main category from 'main_type' because it is always included in the categories from 'type'.
        $categoryData = $this->object['object']['data_map']['type'];
        $categoryIds = explode('#', $categoryData['remote_ids']);

        foreach ($categoryIds as $categoryId) {
            $name = $this->categoryMap->getName($categoryId);
            // New categories might not be in the static category map, yet.
            // We just skip these.
            /** @todo add notification */
            if ($name !== '') {
                $categories[] = new CategoryAdapter(
                    [
                        'remote_id' => $categoryId,
                        'name' => $name,
                    ]
                );
            }
        }

        return $categories;
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia()
    {
        $media = [];
        if (!is_array($this->object['object']['data_map']['gallery']['content'])) {
            return $media;
        }

        foreach ($this->object['object']['data_map']['gallery']['content'] as $key => $image) {
            // Skip importing media records without a source uri.
            if (!empty($image['reference'])) {
                // The array key defines the sorting order which must be preserved.
                $image['_sorting'] = (int) $key;
                $media[] = (new MediumAdapter($image));
            }
        }
        return $media;
    }

    public function hasAttributes()
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return array<AttributeAdapter>
     */
    public function getAttributes()
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    /**
     * @return array
     */
    public function getFiles()
    {
        // TODO
        return [];
    }

    /**
     * @return string|null
     */
    public function getFacebookUri()
    {
        if (!is_array($this->object['object']['data_map']['facebook'])) {
            return null;
        }

        $value = $this->object['object']['data_map']['facebook']['content'];
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return null;
        }

        return $value;
    }

    /**
     * @return string
     */
    public function getTwitterUri()
    {
        if (!is_array($this->object['object']['data_map']['twitter'])) {
            return '';
        }

        $value = $this->object['object']['data_map']['twitter']['content'];
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return '';
        }

        return $value;
    }

    /**
     * @return string
     */
    public function getInstagramUri()
    {
        if (!is_array($this->object['object']['data_map']['instagram'])) {
            return '';
        }

        $value = $this->object['object']['data_map']['instagram']['content'];
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return '';
        }

        return $value;
    }

    /**
     * @return string
     */
    public function getYoutubeUri()
    {
        if (!is_array($this->object['object']['data_map']['youtube'])) {
            return '';
        }

        $value = $this->object['object']['data_map']['youtube']['content'];
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return '';
        }

        return $value;
    }

    /**
     * @return string
     */
    public function getSourceName()
    {
        return '';
    }

    /**
     * @return string|null
     */
    public function getAuthorName()
    {
        if (!empty($this->object['object']['author'])) {
            return $this->object['object']['author'];
        }
        return null;
    }

    /**
     * @return string
     */
    public function getBookingUri()
    {
        return '';
    }

    /**
     * @return string|null
     */
    public function getDetailUri()
    {
        return null;
    }

    /**
     * @param string $value
     * @return bool
     */
    protected function isUrlPlaceholder($value)
    {
        return ($value === 'http://' || $value === 'https://');
    }

    /**
     * The complete opening times data.
     *
     * @return string
     */
    public function getOpeningTimes()
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getLevelOfMaintenance(): int
    {
        return (int) $this->object['object']['data_map']['level']['content'];
    }

    /**
     * Return the star classification for hotels.
     *
     * @return int
     */
    public function getStarClassification()
    {
        return 0;
    }

    /**
     * Return the average rating on a scale of 0-100
     *
     * @return int
     */
    public function getAverageRating()
    {
        return 0;
    }

    /**
     * Return the number of ratings.
     *
     * @return int
     */
    public function getNumberOfRatings(): int
    {
        return 0;
    }

    /**
     * @return array
     */
    private function parseAttributes(): array
    {
        $attributes = [];

        $attributes[PointOfInterestAttributes::FREE_ENTRANCE] =
            $this->object['object']['data_map']['prices_free']['content'];

        if (!empty($this->object['object']['data_map']['priceinfotext']['content'])) {
            $attributes[PointOfInterestAttributes::PRICE_INFORMATION] =
                $this->object['object']['data_map']['priceinfotext']['content'];
        }
        if (!empty($this->object['object']['data_map']['prices']['content'])) {
            $attributes[PointOfInterestAttributes::PRICES] =
                $this->object['object']['data_map']['prices']['content'];
        }
        if (!empty($this->object['object']['data_map']['discount_card']['content'])) {
            $attributes[PointOfInterestAttributes::DISCOUNT_CARD] =
                $this->object['object']['data_map']['discount_card']['content'];
        }

        return $attributes;
    }
}
