<?php
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Constants\Language;
use Newland\Toubiz\Api\Exception\InvalidJsonException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\EventApiService\EventAdapter;

/**
 * Service for legacy Toubiz event API.
 *
 * Concrete implementation to communicate with
 * the first, "old" toubiz API returning events.
 */
class EventApiService extends AbstractService
{
    use LanguageAware;
    use Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://api.toubiz.de/json/';

    /**
     * @var array Object definition for this service.
     */
    const OBJECTS = [
        'Event' => EventAdapter::class,
    ];

    /**
     * Map of language constants to URI segments expected by the API.
     *
     * @var array
     */
    private static $languageMap = [
        Language::DE => 'de',
        Language::EN => 'en',
        Language::FR => 'fr',
        Language::ES => 'es',
    ];

    /**
     * @var string Legacy-specific organizer filter.
     */
    protected $organizerId;

    /**
     * Sets the organizer restriction.
     *
     * @api
     * @param string $organizerId
     * @return void
     */
    public function setOrganizerId($organizerId)
    {
        $this->organizerId = $organizerId;
    }

    /**
     * Fetch active events.
     *
     * @param callable $block
     * @return void
     */
    public function fetchActiveEvents(callable $block)
    {
        $this->fetchEventList(
            function (EventAdapter $listData) use ($block) {
                $details = $this->fetchEventDetails($listData->getExternalId());
                if ($details) {
                    $block($details);
                }
            }
        );
    }

    public function fetchEventList(callable $block)
    {
        $itemsPerPage = 25;
        $offset = 0;

        do {
            try {
                $response = $this->sendRequest(
                    'event/json_result.php',
                    [
                        'offset' => $offset,
                        'per_page' => $itemsPerPage,
                    ]
                );
            } catch (InvalidJsonException $e) {
                throw new InvalidJsonException(
                    sprintf('API request returned invalid JSON (offset %d, per_page %d).', $offset, $itemsPerPage),
                    1557149751
                );
            }
            foreach ($response['result'] ?? [] as $result) {
                $adapter = new EventAdapter($result);
                if ($this->language) {
                    $adapter->setLanguage($this->language);
                }
                $block($adapter);
            }
            $offset += $itemsPerPage;
        } while ($offset <= $response['result_count'] && $this->withinLimit($offset));
    }

    /**
     * Fetch details about the given event identifier.
     *
     * @param string $identifier
     * @return EventAdapter|null
     */
    public function fetchEventDetails(string $identifier)
    {
        $response = $this->sendRequest('event/json_detail.php', [ 'id' => $identifier ]);
        if ($response) {
            $adapter = new EventAdapter($response);
            if ($this->language) {
                $adapter->setLanguage($this->language);
            }
            return $adapter;
        }
        return null;
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param string $path Relative path to self::BASE_URI to call
     * @param array $parameters Optional request parameters
     * @return mixed
     */
    protected function sendRequest(string $path, array $parameters = [])
    {
        $parameters = array_merge(
            $parameters,
            [
                'key' => $this->apiKey,
                'mandant' => $this->clientName,
                'organizer' => $this->organizerId,
                'language' => static::$languageMap[$this->language] ?? 'de',
            ]
        );

        $response = $this->httpClient->request('GET', $path, [ 'query' => $parameters ]);
        if ($response->getStatusCode() === 200) {
            $dataRaw = $response->getBody();
            $dataRaw = StringCleaner::asString($dataRaw);
            $data = json_decode($dataRaw, true);

            if ($data === null) {
                throw new InvalidJsonException(
                    'API request returned invalid JSON.',
                    1557149751
                );
            }

            return $data;
        }

        return false;
    }
}
