<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Tportal\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\LodgingAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Lodging adapter.
 *
 * This represents an Article with mapping for the Tportal lodging.
 *
 * @Flow\Scope("singleton")
 */
class LodgingAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_LODGING;
    }

    public function getExternalId(): string
    {
        return (string) $this->object['housedata']['out']['ID'];
    }

    public function getName(): string
    {
        $name = $this->object['housedata']['out']['sortName'];
        if (empty($name)) {
            $name = $this->object['housedata']['out']['address']['personFirstName'];
            $name .= ' ';
            $name .= $this->object['housedata']['out']['address']['personLastName'];
        }
        if (empty(trim($name))) {
            $name = $this->object['housedata']['out']['describingName'];
        }

        // The name may end with a comma or a dash.
        $suffix = substr($name, -1, 1);
        if ($suffix == ',' || $suffix == '-') {
            $name = substr($name, 0, -1);
        }

        return $name;
    }

    public function getMainAddress(): AddressAdapterInterface
    {
        $address = $this->object;
        $address['name'] = $this->getName();
        return new LodgingAddressAdapter($address);
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getDescription(): ?string
    {
        $part = $this->object['housedata']['out']['descriptions'];
        if ($part) {
            if (array_key_exists('description', $part)) {
                return $part['description'];
            }

            $descriptionPart = $this->findInArray(
                $part,
                'descriptionTypeName',
                'Beschreibung'
            );

            if ($descriptionPart && !empty($descriptionPart['description'])) {
                return $descriptionPart['description'];
            }
        }

        return null;
    }


    /**
     * @return array<CategoryAdapter>
     */
    public function getCategories(): array
    {
        if (empty($this->object['housedata']['out']['touristicType'])) {
            return [];
        }

        return [
            (new CategoryAdapter($this->object['housedata']['out']['touristicType'])),
        ];
    }

    public function getAddresses(): array
    {
        return [];
    }

    /**
     * @return array<MediumAdapter>
     */
    public function getMedia(): array
    {
        if (!empty($this->object['housedata']['out']['multimediaURLs'])) {
            foreach ($this->object['housedata']['out']['multimediaURLs'] as $item) {
                /*
                 * Right now, only the preview image is being imported as there is
                 * no show view for lodgings.
                 *
                 * Next to that, $item can also be a non-array wit useless
                 * information (like a string), or an array with an unexpected
                 * structure that does not provide a complete image.
                 */
                if (is_array($item)
                    && array_key_exists('type', $item)
                    && $item['type']['value'] == 'TP_Preview'
                ) {
                    return [ (new MediumAdapter($item)) ];
                }
            }
        }

        /*
         * And for the case where there is no 'TP_Preview' despite that every
         * record basically must have one, there is 'previewImageURL' but with
         * an already cropped image.
         */
        $url = $this->object['searchresult']['detailInfo']['previewImageURL'];
        if (!empty($item['url'])) {
            $item = [
                'ID' => md5($url),
                'url' => $url,
            ];
            return [ (new MediumAdapter($item)) ];
        }

        return [];
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    public function getFiles(): array
    {
        return [];
    }

    public function getFacebookUri(): ?string
    {
        return null;
    }

    public function getTwitterUri(): ?string
    {
        return null;
    }

    public function getInstagramUri(): ?string
    {
        return null;
    }

    public function getYoutubeUri(): ?string
    {
        return null;
    }

    public function getSourceName(): ?string
    {
        return null;
    }

    public function getAuthorName(): ?string
    {
        return null;
    }

    public function getBookingUri(): ?string
    {
        return '';
    }

    public function getDetailUri(): ?string
    {
        return $this->object['housedata']['out']['niceUrl'];
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getStarClassification(): ?int
    {
        $attributes = $this->object['housedata']['out']['dataClassAttributeValues'];
        $starAttributes = array_values(
            array_filter(
                $attributes,
                function (array $attribute) {
                    return is_array($attribute)
                        && array_key_exists('twsDataClassAttribute', $attribute)
                        && is_array($attribute['twsDataClassAttribute'])
                        && array_key_exists('name', $attribute['twsDataClassAttribute'])
                        && $attribute['twsDataClassAttribute']['name'] === 'ClassificationServiceProvider';
                }
            )
        );

        if (!empty($starAttributes)) {
            // @todo: more than 1 star classification might be present,
            // e.g. for multiple appartments within 1 appartment complex
            $classificationString = $starAttributes[0]['twsLookupValues']['value'];
            // parse the int value of rating string like "5 Sterne Superior"
            if (preg_match('/^(\d)\s.*/', $classificationString, $matches)) {
                return (int) $matches[1];
            }
        }

        return null;
    }

    public function getAverageRating(): ?int
    {
        $rating = $this->object['searchresult']['serviceProviderRatingTotal']['averageGrade'] ?? null;
        if ($rating !== null) {
            return (int) $rating * 10;
        }
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        $numberOfRatings = $this->object['searchresult']['serviceProviderRatingTotal']['numberOfRatings'] ?? null;
        if ($numberOfRatings !== null) {
            return (int) $numberOfRatings;
        }
        return null;
    }

    private function parseAttributes(): array
    {
        $attributes = [];

        if (!empty($this->object['searchresult_details']['services'])) {
            $currency = $this->object['searchresult_details']['services'][0]['minimumBasePrice']['currency'];
            $unit = $this->object['searchresult_details']['services'][0]['minimumBasePriceCalculationRule'];
            $value = $this->object['searchresult_details']['services'][0]['minimumBasePrice']['value'];

            if ($currency && $unit && $value) {
                $attributes[LodgingAttributes::MINIMUM_BASE_PRICE] = [
                    LodgingAttributes::PRICE_CURRENCY => $currency,
                    LodgingAttributes::PRICE_UNIT => $unit,
                    LodgingAttributes::PRICE_VALUE => $value,
                ];
            }
        }
        return $attributes;
    }

    public function getWikipediaUri(): ?string
    {
        return null;
    }

    public function getFlickrUri(): ?string
    {
        return null;
    }
}
