<?php
namespace Newland\Toubiz\Api\Service\Tportal\ObjectAdapter;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Neos\Flow\Annotations as Flow;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\CongressLocationAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Lodging adapter.
 *
 * This represents an Article with mapping for the Tportal lodging.
 *
 * @Flow\Scope("singleton")
 */
class CongressLocationAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    private const FEATURE_GROUP_ROOMS = 'DEU00000060500202372';
    private const FEATURE_NUMBER_OF_ROOMS = 'DEU00000060000287552';
    private const FEATURE_GROUP_NUMBER_OF_ATTENDEES = 'DEU00000060500202512';
    private const FEATURE_NUMBER_OF_ATTENDEES_50 = 'DEU00000060500202479';
    private const FEATURE_NUMBER_OF_ATTENDEES_100 = 'DEU00000060578301418';
    private const FEATURE_NUMBER_OF_ATTENDEES_300 = 'DEU00000060500202506';
    private const FEATURE_NUMBER_OF_ATTENDEES_300_PLUS = 'DEU00000060500202511';

    private static $featureOrderNumberOfAttendees =
        [
            self::FEATURE_NUMBER_OF_ATTENDEES_50,
            self::FEATURE_NUMBER_OF_ATTENDEES_100,
            self::FEATURE_NUMBER_OF_ATTENDEES_300,
            self::FEATURE_NUMBER_OF_ATTENDEES_300_PLUS,
        ];

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_CONGRESS_LOCATION;
    }

    /**
     * @return string
     */
    public function getExternalId(): string
    {
        return (string) $this->object['housedata']['out']['ID'];
    }

    public function getName(): string
    {
        $name = $this->object['housedata']['out']['sortName'];
        if (empty($name)) {
            $name = $this->object['housedata']['out']['address']['personFirstName'];
            $name .= ' ';
            $name .= $this->object['housedata']['out']['address']['personLastName'];
        }
        if (empty(trim($name))) {
            $name = $this->object['housedata']['out']['describingName'];
        }

        // The name may end with a comma or a dash.
        $suffix = $name[strlen($name) - 1];
        if ($suffix === ',' || $suffix === '-') {
            $name = substr($name, 0, -1);
        }

        return $name;
    }

    public function getMainAddress(): AddressAdapterInterface
    {
        $address = $this->object;
        $address['name'] = $this->getName();
        return new CongressLocationAddressAdapter($address);
    }

    public function getDescription(): ?string
    {
        $part = $this->object['housedata']['out']['descriptions'];
        if ($part) {
            if (array_key_exists('description', $part)) {
                return $part['description'];
            }

            $descriptionPart = $this->findInArray(
                $part,
                'descriptionTypeName',
                'Beschreibung'
            );

            if ($descriptionPart && !empty($descriptionPart['description'])) {
                return $descriptionPart['description'];
            }
        }

        return null;
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        if (empty($this->object['housedata']['out']['touristicType'])) {
            return [];
        }

        return [
            new CategoryAdapter($this->object['housedata']['out']['touristicType']),
        ];
    }


    /**
     * @return array
     */
    public function getAddresses(): array
    {
        return [];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        if (!empty($this->object['housedata']['out']['multimediaURLs'])) {
            foreach ($this->object['housedata']['out']['multimediaURLs'] as $item) {
                /*
                 * Right now, only the preview image is being imported as there is
                 * no show view for lodgings.
                 *
                 * Next to that, $item can also be a non-array wit useless
                 * information (like a string), or an array with an unexpected
                 * structure that does not provide a complete image.
                 */
                if (is_array($item)
                    && array_key_exists('type', $item)
                    && $item['type']['value'] === 'TP_Preview'
                ) {
                    return [ new MediumAdapter($item) ];
                }
            }
        }

        /*
         * And for the case where there is no 'TP_Preview' despite that every
         * record basically must have one, there is 'previewImageURL' but with
         * an already cropped image.
         */
        $url = $this->object['searchresult']['detailInfo']['previewImageURL'];
        if (!empty($item['url'])) {
            $item = [
                'ID' => md5($url),
                'url' => $url,
            ];
            return [ (new MediumAdapter($item)) ];
        }

        return [];
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    /**
     * @return string|null
     */
    public function getDetailUri(): ?string
    {
        return $this->object['housedata']['out']['niceUrl'];
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    public function getStarClassification(): int
    {
        $classification = 0;

        $attributes = $this->object['housedata']['out']['dataClassAttributeValues'];
        $starAttributes = array_values(
            array_filter(
                $attributes,
                function (array $attribute) {
                    return is_array($attribute)
                        && array_key_exists('twsDataClassAttribute', $attribute)
                        && is_array($attribute['twsDataClassAttribute'])
                        && array_key_exists('name', $attribute['twsDataClassAttribute'])
                        && $attribute['twsDataClassAttribute']['name'] === 'ClassificationServiceProvider';
                }
            )
        );

        if (!empty($starAttributes)) {
            // @todo: more than 1 star classification might be present,
            // e.g. for multiple appartments within 1 appartment complex
            $classificationString = $starAttributes[0]['twsLookupValues']['value'];
            // parse the int value of rating string like "5 Sterne Superior"
            if (preg_match('/^(\d)\s.*/', $classificationString, $matches)) {
                $classification = (int) $matches[1];
            }
        }

        return $classification;
    }

    public function getAverageRating(): int
    {
        return (int) $this->object['searchresult']['serviceProviderRatingTotal']['averageGrade'] * 10;
    }

    public function getNumberOfRatings(): int
    {
        return (int) $this->object['searchresult']['serviceProviderRatingTotal']['numberOfRatings'];
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $attributes = array_merge($attributes, $this->parseAttributesRoom());
        $attributes = array_merge($attributes, $this->parseAttributesAttendees());

        return $attributes;
    }

    private function parseAttributesRoom(): array
    {
        $attributes = [];

        $featureGroups = $this->object['housedata']['out']['featureGroups'] ?? null;

        if ($featureGroups === null) {
            return $attributes;
        }

        $featureGroupRooms = $this->findInArray(
            $featureGroups,
            'ID',
            self::FEATURE_GROUP_ROOMS
        );

        if (\is_array($featureGroupRooms)) {
            foreach ($featureGroupRooms['features'] as $feature) {
                if (($feature['value']['id'] ?? null) === self::FEATURE_NUMBER_OF_ROOMS) {
                    $attributes[] = new AttributeAdapter(
                        $this->getExternalId(),
                        CongressLocationAttributes::NUMBER_OF_ROOMS,
                        $feature['number']
                    );
                    break;
                }
            }
        }

        return $attributes;
    }

    private function parseAttributesAttendees(): array
    {
        $featureGroups = $this->object['housedata']['out']['featureGroups'] ?? null;

        if ($featureGroups === null) {
            return [];
        }

        $featureGroupNumberOfAttendees = $this->findInArray(
            $featureGroups,
            'ID',
            self::FEATURE_GROUP_NUMBER_OF_ATTENDEES
        );

        if (isset($featureGroupNumberOfAttendees['features']['value']['value'])) {
            return [
                new AttributeAdapter(
                    $this->getExternalId(),
                    CongressLocationAttributes::NUMBER_OF_ATTENDEES,
                    $featureGroupNumberOfAttendees['features']['value']['value']
                ),
            ];
        }

        // If multiple numbers are available, we use the highest.
        if (isset($featureGroupNumberOfAttendees['features'][0])) {
            $order = self::$featureOrderNumberOfAttendees;
            usort(
                $featureGroupNumberOfAttendees['features'],
                function (array $featureA, array $featureB) use ($order) {
                    return (int) array_search($featureB['value']['ID'], $order, true) -
                        (int) array_search($featureA['value']['ID'], $order, true);
                }
            );

            return [
                new AttributeAdapter(
                    $this->getExternalId(),
                    CongressLocationAttributes::NUMBER_OF_ATTENDEES,
                    $featureGroupNumberOfAttendees['features'][0]['value']['value']
                ),
            ];
        }

        return [];
    }


    public function getFiles(): array
    {
        return [];
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getFacebookUri(): ?string
    {
        return null;
    }

    public function getTwitterUri(): ?string
    {
        return null;
    }

    public function getInstagramUri(): ?string
    {
        return null;
    }

    public function getYoutubeUri(): ?string
    {
        return null;
    }

    public function getSourceName(): ?string
    {
        return null;
    }

    public function getAuthorName(): ?string
    {
        return null;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getWikipediaUri(): ?string
    {
        return null;
    }

    public function getFlickrUri(): ?string
    {
        return null;
    }
}
