<?php
namespace Newland\Toubiz\Api\Service\Tportal;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\CongressLocationAdapter;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\LodgingAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\OfferAdapter;
use Newland\Toubiz\Api\Service\Tportal\ObjectAdapter\PackageAdapter;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Newland\Toubiz\Api\Utility\RetryPool;
use Psr\Log\LoggerAwareTrait;

/**
 * Service for legacy Toubiz TPortal.
 *
 * Concrete implementation to communicate with the TPortal
 * which is providing data for TOMAS-bound entities.
 */
class ApiService extends AbstractService
{
    use LoggerAwareTrait;
    use Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://tportal.toubiz.de';

    /**
     * @var array Object definition for this service.
     * @todo: What is this?
     */
    const OBJECTS = [
        'Lodging' => LodgingAdapter::class,
    ];

    public function fetchLodgings(callable $block): void
    {
        $page = 1;
        // By default a page has 10 results.
        while ($this->withinLimit($page * 10)) {
            $data = $this->sendLodgingRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['housedata'] as $id => $item) {
                $lodging = new LodgingAdapter(
                    [
                        'housedata' => $data['housedata'][$id],
                        'searchresult' => $data['searchresult'][$id][0],
                        'searchresult_details' => $data['searchresult_details'][$id],
                    ]
                );
                $block($lodging);
            }

            $page++;
        }
    }

    public function fetchCongressLocations(callable $block): void
    {
        $page = 1;
        while (true) {
            $data = $this->sendCongressLocationRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['housedata'] as $id => $item) {
                $congressLocation = new CongressLocationAdapter(
                    [
                        'housedata' => $data['housedata'][$id],
                        'searchresult' => $data['searchresult'][$id][0],
                        'searchresult_details' => $data['searchresult_details'][$id],
                    ]
                );
                $block($congressLocation);
            }

            $page++;
        }
    }

    public function fetchOffers(callable $block): void
    {
        $page = 1;

        // By default a page has 10 results.
        while ($this->withinLimit($page * 10)) {
            $data = $this->sendOfferRequest($page);
            if (!$data) {
                return;
            }

            // If there is only one result then `offerData` is not an array of
            // data objects but the data object itself.
            if (ArrayUtility::isAssociative($data['offerData'])) {
                $data['offerData'] = [ $data['offerData'] ];
            }

            foreach ($data['offerData'] as $offer) {
                $id = $offer['serviceID'];
                $item = new OfferAdapter(
                    [
                        'offer' => $offer,
                        'serviceData' => $data['serviceData'][$id],
                        'availability' => $data['ServiceAvailabilities'][$id] ?? [],
                        'geoResult' => $this->extractGeoResult($data['geoResultData'] ?? [], $id),
                    ]
                );
                $block($item);
            }

            $page++;
        }
    }

    /**
     * @param array $geoResultData
     * @param string $id
     * @return array|null
     */
    private function extractGeoResult(array $geoResultData, string $id)
    {
        foreach ($geoResultData as $data) {
            if (($data['serviceID'] ?? null) === $id) {
                return $data;
            }
        }
        return null;
    }


    private function sendOfferRequest(int $page)
    {
        $parameters = [
            ':clientName' => $this->clientName,
            ':page' => $page,
        ];

        $url = str_replace(
            array_keys($parameters),
            $parameters,
            '/:clientName/offer?reset=1&json=1&page=:page'
        );

        $response = $this->httpClient->request('GET', $url);
        if ($response->getStatusCode() === 200) {
            $data = json_decode($response->getBody(), true);
            if (is_array($data)
                && array_key_exists('offerData', $data)
                && count($data['offerData']) > 0
            ) {
                return $data;
            }

            return null;
        }

        return null;
    }

    public function fetchPackages(callable $block)
    {
        $page = 1;

        while (true) {
            $data = $this->sendPackageRequest($page);
            if (!$data) {
                return;
            }

            foreach ($data['PackageAccommodationInfos'] as $package) {
                $id = $package['packageID'];
                $item = new PackageAdapter(
                    [
                        'package' => $package,
                        'serviceData' => $data['serviceData'][$id],
                    ]
                );
                $block($item);
            }

            $page++;
        }
    }

    private function sendPackageRequest(int $page)
    {
        $url = $this->urlTemplate(
            '/:clientName/package?reset=1&json=1&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );

        $response = $this->httpClient->request('GET', $url);
        $data = $response->getStatusCode() === 200 ? json_decode($response->getBody(), true) : [];

        if (array_key_exists('PackageAccommodationInfos', $data) && \count($data['PackageAccommodationInfos']) > 0) {
            return $data;
        }

        return null;
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @param int $retriesIfFail
     * @return array|bool
     */
    protected function sendLodgingRequest($page, $retriesIfFail = 5)
    {
        $url = $this->urlTemplate(
            '/:clientName/ukv/search?reset=1&json=1&ukv_result_order=3&page=:page',
            [ ':clientName' => $this->clientName, ':page' => $page ]
        );

        $pool = new RetryPool($retriesIfFail);
        $pool->setLogger($this->logger);
        $response = $pool->retryOnException(
            function () use ($url) {
                return $this->httpClient->request('GET', $url);
            }
        );

        if ($response->getStatusCode() === 200) {
            $data = json_decode($response->getBody(), true);

            /*
             * Checking on housedata if data is present but this could
             * also be any other field that is being returned in the
             * array, as data is spread across multiple response segments.
             */
            if ($data['housedata'] && count($data['housedata']) > 0) {
                return $data;
            }

            return false;
        }

        return false;
    }

    /**
     * Send request to endpoint.
     *
     * This combines request parameters with required
     * authentication parameters and checks the response.
     *
     * @param int $page
     * @param int $retriesIfFail
     * @return array|bool
     */
    protected function sendCongressLocationRequest($page, $retriesIfFail = 5)
    {
        $url = $this->urlTemplate(
            '/:clientName/congress?reset=1&json=1&page=:page',
            [
                ':clientName' => $this->clientName,
                ':page' => $page,
            ]
        );

        $pool = new RetryPool($retriesIfFail);
        $pool->setLogger($this->logger);
        $response = $pool->retryOnException(
            function () use ($url) {
                return $this->httpClient->request('GET', $url);
            }
        );

        if ($response->getStatusCode() === 200) {
            $data = json_decode($response->getBody(), true);

            /*
             * Checking on housedata if data is present but this could
             * also be any other field that is being returned in the
             * array, as data is spread across multiple response segments.
             */
            if (array_key_exists('housedata', $data) && count($data['housedata']) > 0) {
                return $data;
            }
        }

        return false;
    }

    private function urlTemplate(string $url, array $replacements): string
    {
        return str_replace(
            array_keys($replacements),
            array_map('urlencode', array_values($replacements)),
            $url
        );
    }
}
