<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GastronomyAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;

/**
 * Gastronomy adapter.
 *
 * This represents an Article with mapping for the Tportal-specific gastronomy.
 */
class GastronomyAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_GASTRONOMY;
    }

    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    public function getName(): string
    {
        return (string) ($this->object['name'] ?? '');
    }

    public function getDescription(): ?string
    {
        if (!is_array($this->object['description'])) {
            return null;
        }

        $value = null;
        if (array_key_exists('gastro', $this->object['description'])) {
            $value = $this->object['description']['gastro']['text'];
        } elseif (array_key_exists('gastro_event', $this->object['description'])) {
            $value = $this->object['description']['gastro_event']['text'];
        }

        return $value;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        if (!is_array($this->object['images'])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $media[] = (new MediumAdapter($image));
        }
        return $media;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return AttributeAdapterInterface[]
     */
    public function getAttributes(): array
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
        }

        return $items;
    }

    private function getSocialMediaUri(string $socialMedia): ?string
    {
        if (!\is_array($this->object['social_media'])) {
            return null;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            $socialMedia
        );

        return $data ? $data['value'] : null;
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialMediaUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialMediaUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialMediaUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialMediaUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialMediaUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialMediaUri('flickr');
    }

    /** @return FileAdapter[] */
    public function getFiles(): array
    {
        if (!is_array($this->object['files'])) {
            return [];
        }

        $files = [];
        foreach ($this->object['files'] as $file) {
            $files[] = (new FileAdapter($file));
        }
        return $files;
    }

    public function getSourceName(): ?string
    {
        return $this->object['headline_quellmandant'] ?? null;
    }

    public function getAuthorName(): ?string
    {
        if (!empty($this->object['author'])) {
            return $this->object['author'];
        }
        return null;
    }

    public function getBookingUri(): ?string
    {
        if (!is_array($this->object['property'])
            || !array_key_exists('gastro_lunchgate', $this->object['property'])
            || !is_array($this->object['property']['gastro_lunchgate'])
        ) {
            return null;
        }

        $value = $this->object['property']['gastro_lunchgate'][0]['properties'][0]['value'];

        // This is wrong data due to missing valitations.
        if (stristr($value, '@')) {
            return null;
        }

        return $value;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] === '1');
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    private function extractAttributes($key): ?array
    {
        $items = [];

        // The desired key can be in two different places.
        if (array_key_exists($key, $this->object)) {
            $attributes = $this->object[$key];
        } elseif (is_array($this->object['property'])
            && array_key_exists($key, $this->object['property'])
        ) {
            $attributes = $this->object['property'][$key];
        } else {
            return null;
        }

        // Also, the structure is different for finding the attributes.
        if (is_array($attributes[0])
            && array_key_exists('properties', $attributes[0])
        ) {
            $attributes = $attributes[0]['properties'];
        }

        foreach ($attributes as $entry) {
            /*
             * Most of the time, text is only used. But if it also
             * has a value given, it needs to be concatenated.
             */
            if (!empty($entry['value'])) {
                $items[] = $entry['text'] . ': ' . $entry['value'];
            } else {
                $items[] = $entry['text'];
            }
        }

        if (!empty($items)) {
            return $items;
        }

        return null;
    }

    public function getOpeningTimes(): ?string
    {
        return null;
    }

    private function parseAttributes(): array
    {
        $attributes = [];
        $keys = [
            'gastro_ambient' => GastronomyAttributes::AMBIANCE,
            'gastro_child' => GastronomyAttributes::FAMILIES,
            'gastro_group' => GastronomyAttributes::GROUPS,
            'gastro_kitchen' => GastronomyAttributes::KITCHEN_STYLE,
            'gastro_style' => GastronomyAttributes::GASTRONOMY_STYLE,
            'gastro_person' => GastronomyAttributes::CAPACITY,
            'gastro_payment' => GastronomyAttributes::ACCEPTED_PAYMENTS,
            'gastro_pricesegment' => GastronomyAttributes::PRICE_SEGMENT,
            'gastro_property' => GastronomyAttributes::SERVICES,
            'gastro_language' => GastronomyAttributes::SPOKEN_LANGUAGES,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        return $attributes;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getAbstract(): ?string
    {
        return null;
    }
}
