<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DbService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\DbService\CategoryMap;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Utility\ArrayUtility;

/**
 * Point of interest adapter.
 *
 * This represents an Article from the DB service.
 */
class PointOfInterestAdapter extends AbstractLegacyObjectAdapter implements ArticleAdapterInterface
{
    use LanguageAware;

    /**
     * @var CategoryMap
     */
    private $categoryMap;

    public function __construct($adaptee, CategoryMap $categoryMap)
    {
        parent::__construct($adaptee);
        $this->categoryMap = $categoryMap;
    }

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_ATTRACTION;
    }

    public function getExternalId(): string
    {
        return (string) $this->object['remote_id'];
    }

    public function getName(): string
    {
        return substr($this->object['name'], 0, 240);
    }

    public function getAbstract(): ?string
    {
        $value = $this->object['object']['data_map']['intro']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    public function getDescription(): ?string
    {
        $value = $this->object['object']['data_map']['description']['content'];
        if (!empty($value)) {
            return $value;
        }

        return null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $categories = [];

        // We omit the main category from 'main_type' because it is always included in the categories from 'type'.
        $categoryData = $this->object['object']['data_map']['type'];
        $categoryIds = explode('#', $categoryData['remote_ids']);

        foreach ($categoryIds as $categoryId) {
            $name = $this->categoryMap->getName($categoryId);
            // New categories might not be in the static category map, yet.
            // We just skip these.
            /** @todo add notification */
            if ($name !== '') {
                $categories[] = new CategoryAdapter(
                    [
                        'remote_id' => $categoryId,
                        'name' => $name,
                    ]
                );
            }
        }

        $mainCategoryId = $this->object['object']['data_map']['type_main']['remote_id'];
        $categories = $this->moveMainCategoryToTop($categories, $mainCategoryId);

        return $categories;
    }

    /**
     * @param array $categories
     * @param string $mainCategoryId
     * @return CategoryAdapter[]
     */
    public function moveMainCategoryToTop(array $categories, string $mainCategoryId): array
    {
        usort(
            $categories,
            function (CategoryAdapter $categoryA, CategoryAdapter $categoryB) use ($mainCategoryId) {
                return $categoryA->getExternalId() === $mainCategoryId ? -1 : 1;
            }
        );

        return $categories;
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        $media = [];
        if (!is_array($this->object['object']['data_map']['gallery']['content'])) {
            return $media;
        }

        foreach ($this->object['object']['data_map']['gallery']['content'] as $key => $image) {
            // Skip importing media records without a source uri.
            if (!empty($image['reference'])) {
                // The array key defines the sorting order which must be preserved.
                $image['_sorting'] = (int) $key;
                $media[] = (new MediumAdapter($image));
            }
        }
        return $media;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    /**
     * @return AttributeAdapter[]
     */
    public function getAttributes(): array
    {
        $attributes = $this->parseAttributes();

        $items = [];
        foreach ($attributes as $name => $data) {
            if (\is_array($data)) {
                foreach ($data as $datum) {
                    $items[] = new AttributeAdapter($this->getExternalId(), $name, $datum);
                }
            } else {
                $items[] = new AttributeAdapter($this->getExternalId(), $name, $data);
            }
        }

        return $items;
    }

    public function getFiles(): array
    {
        // TODO
        return [];
    }

    private function getSocialUri(string $datamapItemName): ?string
    {
        $value = $this->object['object']['data_map'][$datamapItemName]['content'] ?? null;
        if (empty($value) || $this->isUrlPlaceholder($value)) {
            return null;
        }

        return (string) $value;
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialUri('flickr');
    }

    public function getSourceName(): ?string
    {
        return null;
    }

    public function getAuthorName(): ?string
    {
        if (!empty($this->object['object']['author'])) {
            return $this->object['object']['author'];
        }
        return null;
    }

    protected function isUrlPlaceholder(string $value): bool
    {
        return ($value === 'http://' || $value === 'https://');
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getLevelOfMaintenance(): int
    {
        return (int) $this->object['object']['data_map']['level']['content'];
    }

    private function parseAttributes(): array
    {
        $attributes = [];

        $attributes[PointOfInterestAttributes::FREE_ENTRANCE] =
            $this->object['object']['data_map']['prices_free']['content'];

        if (!empty($this->object['object']['data_map']['priceinfotext']['content'])) {
            $attributes[PointOfInterestAttributes::PRICE_INFORMATION] =
                $this->object['object']['data_map']['priceinfotext']['content'];
        }
        if (!empty($this->object['object']['data_map']['prices']['content'])) {
            $attributes[PointOfInterestAttributes::PRICES] =
                $this->object['object']['data_map']['prices']['content'];
        }
        if (!empty($this->object['object']['data_map']['discount_card']['content'])) {
            $attributes[PointOfInterestAttributes::DISCOUNT_CARD] =
                $this->object['object']['data_map']['discount_card']['content'];
        }
        if (!empty($this->object['object']['data_map']['tags']['content'])) {
            $attributes[PointOfInterestAttributes::TAG] = ArrayUtility::trimExplode(
                ',',
                $this->object['object']['data_map']['tags']['content']
            );
        }

        return $attributes;
    }

    public function getBookingUri(): ?string
    {
        return null;
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getStarClassification(): ?int
    {
        return null;
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }
}
