<?php
namespace Newland\Toubiz\Api\Service\Outdooractive;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Limitable;
use Newland\Toubiz\Api\Service\Outdooractive\ObjectAdapter\TourAdapter;

/**
 * Service for Outdooractive API.
 */
class ApiService extends AbstractService
{
    use LanguageAware;
    use Limitable;

    /**
     * @var string Base URI of API endpoint.
     */
    const DEFAULT_BASE_URI = 'https://www.outdooractive.com/api/project/';

    /**
     * @var string Path for fetching the tours list.
     */
    const PATH_TOURS_LIST = ':clientName/tours?key=:apiKey&lang=:language';

    /**
     * @var string Path for a tour.
     */
    const PATH_TOUR = ':clientName/oois/:id?key=:apiKey&lang=:language';

    /**
     * Fetches list of tour ids for importing.
     *
     * @param callable $block
     * @return void
     */
    public function fetchTours(callable $block): void
    {
        $uri = $this->prepareUri(self::PATH_TOURS_LIST);
        $response = $this->httpClient->request('GET', $uri);
        if ($response->getStatusCode() === 200) {
            $data = simplexml_load_string($response->getBody());
            if (!$data) {
                return;
            }

            $i = 0;
            foreach ($data as $entry) {
                if ($this->exceedsLimit(++$i)) {
                    break;
                }
                $id = (int) $entry['id'];
                $this->fetchTour($block, $id);
            }
        }
    }

    protected function fetchTour(callable $block, $id)
    {
        $uri = $this->prepareUri(self::PATH_TOUR, [ ':id' => $id ]);
        $response = $this->httpClient->request('GET', $uri);
        if ($response->getStatusCode() === 200) {
            $data = simplexml_load_string($response->getBody());
            $article = new TourAdapter($data);
            if ($this->language !== null) {
                $article->setLanguage($this->language);
            }
            $block($article);
        } else {
            return false;
        }
    }

    protected function prepareUri($uri, $parameters = [])
    {
        $parameters = array_merge(
            $parameters,
            [
                ':apiKey' => $this->apiKey,
                ':clientName' => $this->clientName,
                ':language' => $this->language,
            ]
        );

        return str_replace(
            array_keys($parameters),
            $parameters,
            $uri
        );
    }
}
