<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\ExternalIdAdapter;
use Newland\Toubiz\Api\Service\StringCleaner;
use Newland\Toubiz\Api\Service\WithCacheProperty;
use Newland\Toubiz\Api\Utility\ArrayUtility;
use Ramsey\Uuid\Uuid;

trait LegacyToubizArticleAdapterCommon
{
    use WithCacheProperty;


    /** @var array<string, string> */
    protected static $SERVICES_MAPPING = [

        // EN
        'Ball/function rooms' => GenericArticleAttributes::SERVICES_BALL,
        'billiards' => GenericArticleAttributes::SERVICES_BILLIARD,
        'bowling alley/bowling' => GenericArticleAttributes::SERVICES_BOWLING,
        'Bus parking available' => GenericArticleAttributes::SERVICES_BUS,
        'Catering' => GenericArticleAttributes::SERVICES_CATERING,
        'Darts' => GenericArticleAttributes::SERVICES_DARTS,
        'Dogs welcome' => GenericArticleAttributes::SERVICES_DOG,
        'E-Bike Akku-Ladestation' => GenericArticleAttributes::SERVICES_E_BIKE,
        'Frontcooking' => GenericArticleAttributes::SERVICES_FRONTCOOKING,
        'Groups welcome' => GenericArticleAttributes::SERVICES_GROUPS,
        'Handicapped accessible (restaurant and toilets accessible by wheelchair)' =>
            GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Hiking trails close at hand' => GenericArticleAttributes::SERVICES_HIKING,
        'Home delivery service' => GenericArticleAttributes::SERVICES_HOMESERVICE,
        'Live music' => GenericArticleAttributes::SERVICES_LIVE_MUSIC,
        'All-day dining' => GenericArticleAttributes::SERVICES_NEVER_COLD,
        'Nette Toilette' => GenericArticleAttributes::SERVICES_NICE_WC,
        'Nightlife' => GenericArticleAttributes::SERVICES_NIGHTLIFE,
        'Available accommodation (rooms/holiday apartments)' => GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Farm shop selling own products' => GenericArticleAttributes::SERVICES_OWN_PRODUCTS,
        'Adjacent car park' => GenericArticleAttributes::SERVICES_PARKING,
        'Patio, garden' => GenericArticleAttributes::SERVICES_PATIO,
        'No pets allowed' => GenericArticleAttributes::SERVICES_PETS_NOT_ALLOWED,
        'Caan be reached by public transport' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'We mainly use regional products' => GenericArticleAttributes::SERVICES_REGIONAL,
        'Reservations recommended' => GenericArticleAttributes::SERVICES_RESERVATION,
        'Smokers’ room' => GenericArticleAttributes::SERVICES_SMOKING,
        'Smokers room' => GenericArticleAttributes::SERVICES_SMOKING,
        'table football' => GenericArticleAttributes::SERVICES_TABLE_FOOTBALL,
        'Takeaway' => GenericArticleAttributes::SERVICES_TAKEAWAY,
        'Tearoom' => GenericArticleAttributes::SERVICES_TEAROOM,
        'W-Lan' => GenericArticleAttributes::SERVICES_WIFI,

        // DE
        'Festsäle/Gesellschaftsräume' => GenericArticleAttributes::SERVICES_BALL,
        'Billard' => GenericArticleAttributes::SERVICES_BILLIARD,
        'Kegelbahn/Bowling' => GenericArticleAttributes::SERVICES_BOWLING,
        'Busparkplätze vorhanden' => GenericArticleAttributes::SERVICES_BUS,
        'Dart' => GenericArticleAttributes::SERVICES_DARTS,
        'Hunde willkommen' => GenericArticleAttributes::SERVICES_DOG,
        'Reisegruppen willkommen' => GenericArticleAttributes::SERVICES_GROUPS,
        'Behindertengerecht (Lokal und Toiletten mit Rollstuhl erreichbar)' =>
            GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Wanderwege am Haus vorbei' => GenericArticleAttributes::SERVICES_HIKING,
        'Heimlieferservice' => GenericArticleAttributes::SERVICES_HOMESERVICE,
        'Live Musik' => GenericArticleAttributes::SERVICES_LIVE_MUSIC,
        'Durchgehend warme Küche' => GenericArticleAttributes::SERVICES_NEVER_COLD,
        'Übernachtungsmöglichkeiten (Zimmer/Ferienwohnungen)' => GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Hofladen, Verkauf eigener Produkte' => GenericArticleAttributes::SERVICES_OWN_PRODUCTS,
        'Parkmöglichkeiten am Haus' => GenericArticleAttributes::SERVICES_PARKING,
        'Terrasse, Garten' => GenericArticleAttributes::SERVICES_PATIO,
        'Haustiere nicht erlaubt' => GenericArticleAttributes::SERVICES_PETS_NOT_ALLOWED,
        'Mit öffentlichen Verkehrsmitteln zu erreichen' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'Wir verwenden hauptsächlich regionale Produkte' => GenericArticleAttributes::SERVICES_REGIONAL,
        'Reservierung empfohlen' => GenericArticleAttributes::SERVICES_RESERVATION,
        'Raucherraum' => GenericArticleAttributes::SERVICES_SMOKING,
        'Tischfußball' => GenericArticleAttributes::SERVICES_TABLE_FOOTBALL,
        'Take Away' => GenericArticleAttributes::SERVICES_TAKEAWAY,
        'Tea-Room' => GenericArticleAttributes::SERVICES_TEAROOM,

        // FR
        'Salle pour fêtes/locaux de réunion' => GenericArticleAttributes::SERVICES_BALL,
        'billard' => GenericArticleAttributes::SERVICES_BILLIARD,
        'bolwing' => GenericArticleAttributes::SERVICES_BOWLING,
        'Emplacements de parking pour bus' => GenericArticleAttributes::SERVICES_BUS,
        'Service de traiteur' => GenericArticleAttributes::SERVICES_CATERING,
        'jeu de fléchettes' => GenericArticleAttributes::SERVICES_DARTS,
        'Chiens bienvenus' => GenericArticleAttributes::SERVICES_DOG,
        'Voyages organisés bienvenus' => GenericArticleAttributes::SERVICES_GROUPS,
        'Equipement pour handicapés (restaurant et toilettes accessibles en fauteuil roulant)' =>
            GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Sentiers de randonnée près de l’établissement' => GenericArticleAttributes::SERVICES_HIKING,
        'Sentiers de randonnée près de létablissement' => GenericArticleAttributes::SERVICES_HIKING,
        'Sentiers de randonnée près de l établissement' => GenericArticleAttributes::SERVICES_HIKING,
        'Service de livraison à domicile' => GenericArticleAttributes::SERVICES_HOMESERVICE,
        'Musique live' => GenericArticleAttributes::SERVICES_LIVE_MUSIC,
        'Service continu' => GenericArticleAttributes::SERVICES_NEVER_COLD,
        'Possibilités de logement (chambres/appartements de vacances)' => GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Vente à la ferme' => GenericArticleAttributes::SERVICES_OWN_PRODUCTS,
        'Possibilités de parking devant l’établissement' => GenericArticleAttributes::SERVICES_PARKING,
        'Possibilités de parking devant létablissement' => GenericArticleAttributes::SERVICES_PARKING,
        'Possibilités de parking devant l établissement' => GenericArticleAttributes::SERVICES_PARKING,
        'Terrasse, jardin' => GenericArticleAttributes::SERVICES_PATIO,
        'Animaux domestiques non admis' => GenericArticleAttributes::SERVICES_PETS_NOT_ALLOWED,
        'Accessible avec les transports publics' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'Nous utilisons principalement des produits régionaux' => GenericArticleAttributes::SERVICES_REGIONAL,
        'Réservation recommandée' => GenericArticleAttributes::SERVICES_RESERVATION,
        'Salle pour fumeurs' => GenericArticleAttributes::SERVICES_SMOKING,
        'football de table' => GenericArticleAttributes::SERVICES_TABLE_FOOTBALL,
        'Vente à emporter' => GenericArticleAttributes::SERVICES_TAKEAWAY,
        'Salon de thé' => GenericArticleAttributes::SERVICES_TEAROOM,

        // ES
        'Sala de fiestas/salones privados' => GenericArticleAttributes::SERVICES_BALL,
        'billar' => GenericArticleAttributes::SERVICES_BILLIARD,
        'bolera' => GenericArticleAttributes::SERVICES_BOWLING,
        'Aparcamiento para autocares' => GenericArticleAttributes::SERVICES_BUS,
        'dardos' => GenericArticleAttributes::SERVICES_DARTS,
        'Se aceptan perros' => GenericArticleAttributes::SERVICES_DOG,
        'Adecuado para viajes organizados' => GenericArticleAttributes::SERVICES_GROUPS,
        'Adaptado para minusválidos (comedor y servicios accesibles con silla de ruedas)' =>
            GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Senderos de excursión junto a la casa' => GenericArticleAttributes::SERVICES_HIKING,
        'Vida nocturna' => GenericArticleAttributes::SERVICES_NIGHTLIFE,
        'Posibilidad de pernoctación (habitaciones / apartamentos)' => GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Posibilidad de aparcamiento' => GenericArticleAttributes::SERVICES_PARKING,
        'Terraza, jardín' => GenericArticleAttributes::SERVICES_PATIO,
        'No se permiten animales' => GenericArticleAttributes::SERVICES_PETS_NOT_ALLOWED,
        'Transporte público local cercano' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'Utilizamos principalmente productos regionales' => GenericArticleAttributes::SERVICES_REGIONAL,
        'Reservas recomendado' => GenericArticleAttributes::SERVICES_RESERVATION,
        'Sala de fumadores' => GenericArticleAttributes::SERVICES_SMOKING,
        'futbolín' => GenericArticleAttributes::SERVICES_TABLE_FOOTBALL,

        // IT
        'Saloni per feste o riunioni' => GenericArticleAttributes::SERVICES_BALL,
        'biliardo' => GenericArticleAttributes::SERVICES_BILLIARD,
        'bowling' => GenericArticleAttributes::SERVICES_BOWLING,
        'Parcheggio per pullman' => GenericArticleAttributes::SERVICES_BUS,
        'freccetta' => GenericArticleAttributes::SERVICES_DARTS,
        'Si accettano cani' => GenericArticleAttributes::SERVICES_DOG,
        'Si accettano gruppi' => GenericArticleAttributes::SERVICES_GROUPS,
        'Conforme ai portatori di handicap (sala pranzo e WC raggiungibili in carrozzina)' =>
            GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Posizione lungo sentieri' => GenericArticleAttributes::SERVICES_HIKING,
        'Vita notturna' => GenericArticleAttributes::SERVICES_NIGHTLIFE,
        'Possibilità di pernottamento (camere/appartamenti per vacanze)' =>
            GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Parcheggio' => GenericArticleAttributes::SERVICES_PARKING,
        'Terrazza, giardino' => GenericArticleAttributes::SERVICES_PATIO,
        'Animali domestici non permessi' => GenericArticleAttributes::SERVICES_PETS_NOT_ALLOWED,
        'Raggiungibile con mezzi pubblici' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'Utilizziamo principalmente prodotti regionali' => GenericArticleAttributes::SERVICES_REGIONAL,
        'Prenotazione consigliata' => GenericArticleAttributes::SERVICES_RESERVATION,
        'Sala fumatori' => GenericArticleAttributes::SERVICES_SMOKING,
        'calcetto' => GenericArticleAttributes::SERVICES_TABLE_FOOTBALL,

        'mit öffentlichen Verkehrsmitteln zu erreichen' => GenericArticleAttributes::SERVICES_PUBLIC_TRANSIT,
        'Gläserne Produktion' => GenericArticleAttributes::SERVICES_PUBLICLY_OBSERVABLE_PRODUCTION,
        'Behindertengerecht' => GenericArticleAttributes::SERVICES_HANDICAP_ACESSIBLE,
        'Übernachtungsmöglichkeiten' => GenericArticleAttributes::SERVICES_OVERNIGHT,
        'Produkte aus ökologischem Anbau' => GenericArticleAttributes::SERVICES_ECOLOGICAL_PRODUCTS,
    ];


    public function getSourceSystem(): string
    {
        return self::SOURCE_TOUBIZ_LEGACY;
    }

    protected function setupLegacyArticleAdapterCommon(): void
    {
        $this->stringCleaner = new StringCleaner();
    }

    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['remote_id']);
    }

    public function getLanguageGrouping(): ?string
    {
        // Use toubiz new mapping if exists
        $toubizNewId = $this->object['object']['data_map']['infosystem_id']['content'] ?? null;
        if ($toubizNewId) {
            return $toubizNewId;
        }

        $id = trim(strip_tags((string) ($this->object['object']['id'] ?? '')));
        if ($id) {
            return md5(static::class . 'id' . $id);
        }

        return null;
    }

    public function getName(): string
    {
        return substr(
            strip_tags((string) $this->object['name']),
            0,
            240
        );
    }

    public function getAbstract(): ?string
    {
        return $this->stringCleaner->purifyHtml((string) $this->getDataMapString('intro'));
    }

    public function getDescription(): ?string
    {
        return $this->stringCleaner->purifyHtml((string) $this->getDataMapString('description'));
    }


    public function getLatitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[0];
        }

        $latitude = $this->object['location']['Latitude'] ?? null;
        return $latitude ? (float) $latitude : null;
    }

    public function getLongitude(): ?float
    {
        $coordinates = explode('#', $this->object['geo'] ?? '');
        if (\count($coordinates) === 2) {
            return (float) $coordinates[1];
        }

        $longitude = $this->object['location']['Longitude'] ?? null;
        return $longitude ? (float) $longitude : null;
    }


    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    /**
     * @return bool
     * @deprecated
     */
    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    protected function parseCommonToubizAttributes(): array
    {
        $attributes = [];

        $searchWords = trim($this->object['object']['data_map']['searchwords']['content'] ?? '');
        if ($searchWords) {
            $attributes[GenericArticleAttributes::ADDITIONAL_SEARCH_STRING] = $searchWords;
        }

        $clientName = $this->getSourceName();
        if ($clientName) {
            $attributes[GenericArticleAttributes::CLIENT_NAME] = $clientName;
        }

        $authorName = $this->getAuthorName();
        if ($authorName) {
            $attributes[GenericArticleAttributes::AUTHOR_NAME] = $authorName;
        }

        return $attributes;
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialUri('facebook');
    }

    public function getTwitterUri(): ?string
    {
        return $this->getSocialUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialUri('flickr');
    }

    public function getSourceName(): ?string
    {
        $source = $this->object['headline_quellmandant'] ?? null;
        return $source ? strip_tags($source) : null;
    }

    public function getAuthorName(): ?string
    {
        $author = $this->object['object']['author'] ?? null;
        return $author ? strip_tags($author) : null;
    }

    protected function getDataMapString(string $dataMapItemName): ?string
    {
        $value = $this->object['object']['data_map'][$dataMapItemName]['content'] ?? null;
        return $value ? (string) $value : null;
    }

    protected function getSocialUri(string $name): ?string
    {
        $value = $this->getDataMapString($name);
        if ($value === null) {
            $socialMedia = $this->object['social_media'] ?? [];
            $property = $this->findInArray(
                $socialMedia,
                'property',
                $name
            );
            $value = $property['value'] ?? null;
        }

        if ($value === null) {
            return null;
        }

        $value = trim($value);

        if ($value === '') {
            return null;
        }

        if (!$this->validateUrl($value)) {
            return null;
        }

        return $value;
    }

    protected function validateUrl(string $url): ?bool
    {
        return (bool) preg_match(
            '/^(?:https?:\/\/)?[\w\.\-]+(?:\.[\w\-\.]+)+[\w\!\#\$\%\&\(\)\*\+\,\-\.\/\:\;\=\?\@\[\]\~]+$/',
            $url
        );
    }


    /**
     * Finds the first matching key/value pair inside an array of arrays.
     *
     * @param array $array
     * @param mixed $searchKey
     * @param mixed $searchValue
     * @return array|null
     * @example
     *      $arr = [
     *          [
     *              "descriptionTypeID" => "",
     *              "descriptionTypeName" => "description"
     *              "description" => "..."
     *          ]
     *      ];
     *      $descriptionArray = $this->findInArray($arr, 'descriptionTypeName', 'description');
     *
     */
    protected function findInArray(array $array, $searchKey, $searchValue)
    {
        $result = array_filter(
            $array,
            function ($item) use ($searchKey, $searchValue) {
                return ((array) $item)[$searchKey] == $searchValue;
            }
        );

        if (count($result)) {
            return (array) array_shift($result);
        }

        return null;
    }

    /**
     * Extracts data map entries into a single associative array.
     * The given entries array must be a mapping of the normalized keys to the
     * ones in the data map.
     *
     * @param array $entries
     * @return array
     */
    protected function mapDataMapEntries(array $entries): array
    {
        $attributes = [];
        foreach ($entries as $normalized => $external) {
            $value = $this->getDataMapEntry($external);
            if ($value !== null) {
                $attributes[$normalized] = $value;
            }
        }
        return $attributes;
    }

    /**
     * @return array|bool|int|string|string[]|null
     */
    protected function getDataMapEntry(?string $name)
    {
        $entry = $this->object['object']['data_map'][$name] ?? null;
        if (!\is_array($entry)) {
            return null;
        }
        return $this->normalizeDataMapItem($entry);
    }

    /**
     * @return array|bool|int|string|string[]|null
     */
    protected function normalizeDataMapItem(array $item)
    {
        $type = $item['datatypestring'];
        $content = $item['content'];

        switch ($type) {
            case 'ezobjectrelationlist':
            case 'ezxmltext':
                return $content ? (string) $content : null;
            case 'ezstring':
                return $content ? strip_tags((string) $content) : null;
            case 'ezselection':
            case 'ezkeyword':
                if (!is_string($content) || $content === '') {
                    return null;
                }
                $items = ArrayUtility::trimExplode(',', $content);
                if (empty($items) || ($items[0] ?? null) === '---') {
                    return null;
                }
                return $items;
            case 'ezinteger':
                return $content ? (int) $content : null;
            case 'ezboolean':
                if ($content === true) {
                    return true;
                }
                if ($content === false) {
                    return false;
                }
                return null;
            case 'ezmatrix':
                if (!is_array($content) || empty($content)) {
                    return null;
                }
                return (array) $content;
            case 'ezimage':
                return $content['reference'] ?? $content['medium'] ?? null;
            case 'ezurl':
                if (in_array($content, [ null, 'http://', 'https://' ], true)) {
                    return null;
                }
                return (string) $content;
        }

        return null;
    }

    protected function extractMediaFromGallery(callable $adapterInstanciator): array
    {
        return $this->cache(
            'media',
            function () use ($adapterInstanciator) {
                $raw = $this->object['object']['data_map']['gallery']['content'] ?? null;
                if (!\is_array($raw)) {
                    return [];
                }

                // Skip importing media records without a source uri.
                $raw = array_filter(
                    $raw,
                    function (array $item) {
                        return !empty($item['reference']);
                    }
                );

                $media = [];
                foreach ($raw as $key => $image) {
                    $media[] = $adapterInstanciator(array_merge($image, [ '_sorting' => (int) $key ]));
                }
                return $media;
            }
        );
    }


    public function getAdditionalExternalIds(): array
    {
        $externalIds = [
            new ExternalIdAdapter(ExternalIdType::TOUBIZ_LEGACY_REMOTE_ID, (string) $this->object['remote_id']),
        ];

        $id = $this->object['object']['id'] ?? null;
        if ($id) {
            $externalIds[] = new ExternalIdAdapter(ExternalIdType::TOUBIZ_LEGACY, (string) $id);
        }

        $dataMapMapping = [
            ExternalIdType::TOMAS => 'tomas_id',
            ExternalIdType::OUTDOORACTIVE => 'alpstein_id',
        ];

        foreach ($dataMapMapping as $type => $dataMapKey) {
            $id = $this->getDataMapString($dataMapKey);
            if ($id) {
                $externalIds[] = new ExternalIdAdapter($type, $id);
            }
        }

        // Sometimes toubiz_id contains toubiz legacy ids and sometimes toubiz new ids.
        // If it is a uuid, then it's toubiz new.
        $toubizId = $this->getDataMapString('toubiz_id');
        if ($toubizId) {
            $type = Uuid::isValid($toubizId) ? ExternalIdType::TOUBIZ : ExternalIdType::TOUBIZ_LEGACY;
            $externalIds[] = new ExternalIdAdapter($type, $toubizId);
        }

        return $externalIds;
    }

    public function getAdditionalSearchString(): ?string
    {
        return null;
    }
}
