<?php
namespace Newland\Toubiz\Sync\Neos\Tests\Unit\Service\Toubiz\Legacy;


use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Middleware;
use GuzzleHttp\Psr7\Response;
use Neos\Flow\Tests\FunctionalTestCase;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\CityApiService;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;

class CityApiServiceTest extends TestCase
{

    /** @var CityApiService */
    private $subject;

    public function setUp(): void
    {
        parent::setUp();
        $this->subject = new CityApiService();
        $this->subject->setLogger(new NullLogger());
        $this->subject->setClientName('__client_name__');
        $this->subject->setApiKey('__api_key__');
        $this->subject->setParameters([ 'concurrency' => 1, 'sleepSecondsOnError' => 0.0 ]);
    }

    public function testFetchesCitiesFromToubiz(): void
    {
        $this->mockNextResponses([
            new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children' => [], 'hidden' => [] ] ])),
            new Response(200, ['Content-Type' => 'application/json'], file_get_contents(__DIR__ . '/../../../../Fixtures/ToubizLegacyData/cityData.json')),
            new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
        ], $history);

        $this->subject->fetchCities(function() {});

        $this->assertCount(3, $history, 'Service should have made 3 requests');

        $this->assertStringContainsString('(getremoteids)', (string) $history[0]['request']->getUri());
        $this->assertStringContainsString('(class)/city', (string) $history[0]['request']->getUri());
        $this->assertStringContainsString('__client_name__', (string) $history[0]['request']->getUri());
        $this->assertStringContainsString('(key)/__api_key__', (string) $history[0]['request']->getUri());

        $this->assertStringContainsString('(class)/city', (string) $history[1]['request']->getUri());
        $this->assertStringContainsString('(key)/__api_key__', (string) $history[1]['request']->getUri());
        $this->assertContains('/(offset)/0', (string) $history[1]['request']->getUri());
        $this->assertContains('/(limit)/25', (string) $history[1]['request']->getUri());

        $this->assertStringContainsString('(class)/city', (string) $history[2]['request']->getUri());
        $this->assertStringContainsString('(key)/__api_key__', (string) $history[2]['request']->getUri());
        $this->assertContains('/(offset)/25', (string) $history[2]['request']->getUri());
        $this->assertContains('/(limit)/25', (string) $history[2]['request']->getUri());
    }

    public function testFetchesListOfAvailableAndDeletedArticles(): void
    {
        $this->mockNextResponses([
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [
                 'children' => [
                     'foo',
                     'bar',
                     'baz',
                 ],
                 'hidden' => [
                     'this_is_hidden',
                     'lalala'
                 ]
             ] ])),
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
         ], $history);

        $result = $this->subject->fetchCities(function() {});
        $this->assertContains('foo', $result->getAll());
        $this->assertContains('bar', $result->getAll());
        $this->assertContains('baz', $result->getAll());
        $this->assertContains('this_is_hidden', $result->getDeleted());
        $this->assertContains('lalala', $result->getDeleted());
    }

    public function testCreatesAdaptersForCities(): void
    {
        $this->mockNextResponses([
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children' => [], 'hidden' => [] ] ])),
             new Response(200, ['Content-Type' => 'application/json'], file_get_contents(__DIR__ . '/../../../../Fixtures/ToubizLegacyData/cityData.json')),
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
        ]);

        $adapters = [];
        $this->subject->fetchCities(function($adapter) use (&$adapters) {
            $adapters[] = $adapter;
            $this->assertInstanceOf(ArticleAdapterInterface::class, $adapter);
        });
        $this->assertGreaterThan(0, \count($adapters), 'The callback should have been called more than once');
    }

    public function testAddsLanguageParameterToRequestUri(): void
    {
        // No language
        $history = [];
        $this->mockNextResponses([
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children' => [], 'hidden' => [] ] ])),
             new Response(200, ['Content-Type' => 'application/json'], file_get_contents(__DIR__ . '/../../../../Fixtures/ToubizLegacyData/cityData.json')),
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
        ], $history);

        $this->subject->fetchCities(function() {});
        $this->assertNotContains('(lang)', (string) $history[1]['request']->getUri());

        // German
        $history = [];
        $this->mockNextResponses([
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children' => [], 'hidden' => [] ] ])),
             new Response(200, ['Content-Type' => 'application/json'], file_get_contents(__DIR__ . '/../../../../Fixtures/ToubizLegacyData/cityData.json')),
            new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
        ], $history);

        $this->subject->setLanguage('de');
        $this->subject->fetchCities(function() {});
        $this->assertContains('/(lang)/ger-DE', (string) $history[1]['request']->getUri());

        // French
        $history = [];
        $this->mockNextResponses([
             new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children' => [], 'hidden' => [] ] ])),
             new Response(200, ['Content-Type' => 'application/json'], file_get_contents(__DIR__ . '/../../../../Fixtures/ToubizLegacyData/cityData.json')),
            new Response(200, ['Content-Type' => 'application/json'], json_encode([ 'content' => [ 'children_list' => [] ] ])),
        ], $history);

        $this->subject->setLanguage('fr');
        $this->subject->fetchCities(function() {});
        $this->assertContains('/(lang)/fre-FR', (string) $history[1]['request']->getUri());

    }

    /**
     * @param $responses Response[]
     * @param array $history
     */
    private function mockNextResponses($responses, &$history = [], int $repeatLastResponse = 15): void
    {
        if ($repeatLastResponse > 0) {
            $lastResponse = array_pop($responses);
            for ($i = 0; $i < $repeatLastResponse; $i++) {
                $responses[] = $lastResponse;
            }
        }

        $history = [];
        $mock = new MockHandler($responses);
        $stack = HandlerStack::create($mock);
        $stack->push(Middleware::history($history));

        $this->subject->setHttpClientSettings([ 'handler' => $stack ]);
    }
}
