<?php
namespace Newland\Toubiz\Api\Tests\Unit\Service\Toubiz\Legacy\Brochure;

use Newland\Toubiz\Api\Service\Toubiz\Legacy\Brochure\BrochureRedirectUriConverter;
use PHPUnit\Framework\TestCase;

class BrochureRedirectUriConverterTest extends TestCase
{

    public function testConvertsSuccessfulRedirects()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(status)/ok');
        $this->assertTrue($result->isSuccess());
    }

    public function testConvertsFailures()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(status)/fail/(errors)/choice_language!Bitte Sprache wÃ¤hlen/(request)/choice_salutation,mr!entry_firstname,Firstname!entry_lastname,Lastname!entry_street,Test street!entry_zip,12345!entry_location,Test City!count,3851!count_pros,3851|1');
        $this->assertFalse($result->isSuccess());
    }

    public function testTreatsSuccessfulRedirectsWithErrorsAsFailures()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(status)/ok/(errors)/choice_language!Bitte Sprache wÃ¤hlen/(request)/choice_salutation,mr!entry_firstname,Firstname!entry_lastname,Lastname!entry_street,Test street!entry_zip,12345!entry_location,Test City!count,3851!count_pros,3851|1');
        $this->assertFalse($result->isSuccess());
    }

    public function testDefaultsToFailureIfNoStatusGiven()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(request)/choice_salutation,mr!entry_firstname,Firstname!entry_lastname,Lastname!entry_street,Test street!entry_zip,12345!entry_location,Test City!count,3851!count_pros,3851|1');
        $this->assertFalse($result->isSuccess());
    }

    public function testExtractsErrors()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(errors)/foo!bar,baz!test');
        $this->assertEquals(
            [ 'foo' => 'bar', 'baz' => 'test' ],
            $result->getErrors()
        );
    }

    public function testExtractsRequestParameters()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(request)/foo,bar!baz,test');
        $this->assertEquals(
            [ 'foo' => 'bar', 'baz' => 'test' ],
            $result->getRequest()
        );
    }

    public function testMarkedFieldsAreTreatedAsBoolean()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(errors)/foo!mark');
        $this->assertTrue($result->getErrors()['foo']);
    }

    public function testExtractsDeepRequestData()
    {
        $result = (new BrochureRedirectUriConverter())->convert('/(request)/foo,bar-baz!foo_pros,bar|val1-baz|val2');
        $this->assertEquals(
            [ 'foo' => [ 'bar' => 'val1', 'baz' => 'val2' ] ],
            $result->getRequest()
        );
    }

    /**
     * @dataProvider getCompleteRealUris
     * @param string $uri
     * @param bool $success
     * @param array $request
     * @param array $errors
     */
    public function testCorrectlyParsesCompleteRealUris(string $uri, bool $success, array $request, array $errors)
    {
        $result = (new BrochureRedirectUriConverter())->convert($uri);
        $this->assertEquals($success, $result->isSuccess());
        $this->assertEquals($request, $result->getRequest());
        $this->assertEquals($errors, $result->getErrors());
    }

    public function getCompleteRealUris(): array
    {
        return [
            [
                // URI
                '/(status)/ok',
                // Success
                true,
                // Request
                [],
                // Errors
                []
            ],
            [
                // URI
                '/(status)/fail/(errors)/entry_firstname!mark,entry_lastname!mark,no_prospekt!Ausflugstipps%20im%20Sauerland/(request)/choice_salutation,mr!entry_street,Test%20street!entry_zip,12345!entry_location,Test%20City!count,3851-999!count_pros,3851%7C1-999%7C55',
                // Success
                false,
                // Request
                [
                    'choice_salutation' => 'mr',
                    'entry_street' => 'Test street',
                    'entry_zip' => '12345',
                    'entry_location' => 'Test City',
                    'count' => [ '3851' => '1', '999' => '55' ]
                ],
                // Errors
                [ 'entry_lastname' => true, 'entry_firstname' => true, 'no_prospekt' => 'Ausflugstipps im Sauerland' ]
            ],
        ];
    }
}
