<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Utility;

class AttributeImportUtility
{

    /**
     * Splits the given associative array of attributes into single importable attribute
     * items and calls the given block for each of them.
     *
     * If the value in the array is a primitive array then each item in that array is
     * imported independently.
     *
     * The following input array
     * ```
     * [
     *      'foo' => 'value1',
     *      'bar' => [ 'value2', 'value3' ],
     *      'baz' => [ [ 'value4', 'value5' ], [ 'value6', 'value7' ],
     * ]
     * ```
     *
     * will call the given block with the following arguments:
     * - 'foo', 'value1'
     * - 'bar', 'value2' (array of primitives)
     * - 'bar', 'value3' (array of primitives)
     * - 'baz', [ [ 'value4', 'value5' ], [ 'value6', 'value7' ] (complex array)
     *
     * @param array $attributes
     * @param callable $block
     * @return array
     */
    public static function splitAttributeArrayIntoImportables(array $attributes, callable $block): array
    {
        $items = [];
        foreach ($attributes as $name => $value) {
            if (empty($value)) {
                continue;
            }
            if (ArrayUtility::isArrayOfPrimitives($value)) {
                foreach ($value as $dataItem) {
                    $item = $block($name, $dataItem);
                    if ($item) {
                        $items[] = $item;
                    }
                }
            } else {
                $item = $block($name, $value);
                if ($item) {
                    $items[] = $item;
                }
            }
        }
        return $items;
    }
}
