<?php declare(strict_types=1);
namespace Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\DirectMarketerApiService;

/*
 * This file is part of the "toubiz-api" package.
 *
 * For the full copyright and license information, please read the
 * LICENSE.txt file that was distributed with this source code.
 */

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ArticleWithSocialMediaLinksAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Article\ExternalIdSelector;
use Newland\Toubiz\Api\ObjectAdapter\ArticleAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\DirectMarketerAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ExternalIdType;
use Newland\Toubiz\Api\ObjectAdapter\HasAdditionalExternalIds;
use Newland\Toubiz\Api\ObjectAdapter\HasLanguageGroupingSeparateFromOriginalId;
use Newland\Toubiz\Api\ObjectAdapter\YoutubeVideoAdapter;
use Newland\Toubiz\Api\ObjectAdapter\VimeoVideoAdapter;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\AbstractLegacyObjectAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\GastronomyApiService\AttributeAdapter;
use Newland\Toubiz\Api\Service\Toubiz\Legacy\ObjectAdapter\LegacyToubizArticleAdapterCommon;
use Newland\Toubiz\Api\Service\UsesFirstMediumAsMainMedium;
use Newland\Toubiz\Api\Utility\AttributeImportUtility;
use Newland\Toubiz\Api\Utility\UrlUtility;
use Psr\Log\LoggerAwareTrait;

/**
 * Direct marketer adapter.
 *
 * This represents an Article with mapping for the Toubiz specific direct marketer.
 */
class DirectMarketerAdapter extends AbstractLegacyObjectAdapter implements
    ArticleAdapterInterface,
    HasLocationDataInterface,
    ArticleWithSocialMediaLinksAdapterInterface,
    HasLanguageGroupingSeparateFromOriginalId,
    HasAdditionalExternalIds
{
    use LanguageAware,
        UsesFirstMediumAsMainMedium,
        LegacyToubizArticleAdapterCommon,
        LoggerAwareTrait;

    // Note: Data from API is always in english or german, no matter the requested language.
    private const SHOPPING_MAPPING = [
        'At the farm' => DirectMarketerAttributes::SHOPPING_FARM,
        'Shop' => DirectMarketerAttributes::SHOPPING_SHOP,
        'Home delivery' => DirectMarketerAttributes::SHOPPING_HOME_DELIVERY,
        'Postal shipping' => DirectMarketerAttributes::SHOPPING_POSTAL_SHIPPING,
        'Market' => DirectMarketerAttributes::SHOPPING_MARKET,

        'Auf dem Hof' => DirectMarketerAttributes::SHOPPING_FARM,
        'Ladengeschäft' => DirectMarketerAttributes::SHOPPING_SHOP,
        'Lieferservice' => DirectMarketerAttributes::SHOPPING_HOME_DELIVERY,
        'Postversand' => DirectMarketerAttributes::SHOPPING_POSTAL_SHIPPING,
        'Wochen- / Bauernmarkt' => DirectMarketerAttributes::SHOPPING_MARKET,
    ];

    // Note: These are named the same in all languages
    private const DISTINCTION_MAPPING = [
        'Bioland' => DirectMarketerAttributes::DISTINCTIONS_BIOLAND,
        'Demeter' => DirectMarketerAttributes::DISTINCTIONS_DEMETER,
        'Ecovin' => DirectMarketerAttributes::DISTINCTIONS_ECOVIN,
        'EU-Bio-Siegel' => DirectMarketerAttributes::DISTINCTIONS_EU_BIO,
        'Naturland' => DirectMarketerAttributes::DISTINCTIONS_NATURLAND,
    ];

    // Note: These are only ever returned in german
    private const MEAT_MAPPING = [
        'Lamm' => DirectMarketerAttributes::MEATS_LAMB,
        'Rind' => DirectMarketerAttributes::MEATS_BEEF,
        'Schwein' => DirectMarketerAttributes::MEATS_PORK,
        'Ziege' => DirectMarketerAttributes::MEATS_GOAT,
        'Wild' => DirectMarketerAttributes::MEATS_DEER,
        'Kaninchen' => DirectMarketerAttributes::MEATS_RABBIT,
    ];

    private const FISH_MAPPING = [
        'Eier' => DirectMarketerAttributes::FISH_EGGS,
        'Ente' => DirectMarketerAttributes::FISH_DUCK,
        'Gans' => DirectMarketerAttributes::FISH_GOOSE,
        'Hähnchen / Hühner' => DirectMarketerAttributes::FISH_CHICKEN,
        'Pute' => DirectMarketerAttributes::FISH_TURKEY,
        'Fisch' => DirectMarketerAttributes::FISH_FISH,
    ];

    private const DRINK_MAPPING = [
        'Saft / Sirup' => DirectMarketerAttributes::DRINKS_JUICE,
        'Sekt' => DirectMarketerAttributes::DRINKS_SPARKLING_WINE,
        'Schnaps / Brände / Liköre' => DirectMarketerAttributes::DRINKS_LIQUOR,
        'Wein' => DirectMarketerAttributes::DRINKS_WINE,
        'Bier' => DirectMarketerAttributes::DRINKS_BEER,
        'Most' => DirectMarketerAttributes::DRINKS_FRUIT_WINE,
    ];

    private const VEGETABLE_MAPPING = [
        'Äpfel' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_APPLE,
        'Kartoffeln' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_POTATO,
        'Beeren-/Steinobst' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_BERRY,
        'Birnen' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_PEAR,
        'Eingelegtes Gemüse / Obst' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_PICKLED_VEGETABLES,
        'Kräuter' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_HERBS,
        'Gemüse / Salat' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_SALAD,
        'Gewürze / Tee' => DirectMarketerAttributes::VEGETABLES_AND_FRUIT_TEA,
    ];

    private const OIL_MAPPING = [
        'Honig' => DirectMarketerAttributes::OILS_SPREADS_HONEY,
        'Öl / Essig' => DirectMarketerAttributes::OILS_SPREADS_OIL_AND_VINEGAR,
        'Marmelade / Gelees' => DirectMarketerAttributes::OILS_SPREADS_JAM,
        'Brotaufstriche' => DirectMarketerAttributes::OILS_SPREADS_SPREAD,
    ];

    private const MILK_MAPPING = [
        'Eis' => DirectMarketerAttributes::MILK_ICE_CREAM,
        'Käse' => DirectMarketerAttributes::MILK_SHEEP,
        'Frischmilch' => DirectMarketerAttributes::MILK_MILK,
        'Quark' => DirectMarketerAttributes::MILK_QUARK,
        'Ziegenprodukte' => DirectMarketerAttributes::MILK_GOAT,
        'Joghurt' => DirectMarketerAttributes::MILK_YOGURT,
    ];

    private const GRAIN_MAPPING = [
        'Nudeln' => DirectMarketerAttributes::GRAINS_NOODLES,
        'Brot / Brötchen / Gebäck' => DirectMarketerAttributes::GRAINS_BREAD,
        'Nüsse' => DirectMarketerAttributes::GRAINS_NUTS,
        'Getreide / Mehl' => DirectMarketerAttributes::GRAINS_FLOUR,
        'Hülsenfrüchte' => DirectMarketerAttributes::GRAINS_LEGUMES,
        'Birnbrot' => DirectMarketerAttributes::GRAINS_PEAR_BREAD,
    ];

    private const SEASONALS_MAPPING = [
        'Kürbis' => DirectMarketerAttributes::SEASONALS_PUMPKIN,
        'Weihnachtsbäume/-schmuck' => DirectMarketerAttributes::SEASONALS_CHRISTMAS_DECORATION,
        'Erdbeeren' => DirectMarketerAttributes::SEASONALS_STRAWBERRIES,
        'Spargel' => DirectMarketerAttributes::SEASONALS_ASPARGUS,
        'Neuer Süßer' => DirectMarketerAttributes::SEASONALS_SWEET_WINE,
    ];

    private const CRAFTS_MAPPING = [
        'Felle' => DirectMarketerAttributes::CRAFTS_FUR,
        'Kosmetik / Seife' => DirectMarketerAttributes::CRAFTS_COSMETICS,
        'Blumen / Floristik' => DirectMarketerAttributes::CRAFTS_FLOWERS,
        'Deko-/ Geschenkartikel' => DirectMarketerAttributes::CRAFTS_DECORATIONS,
        'Glaswaren / Glaskunst' => DirectMarketerAttributes::CRAFTS_GLASS,
        'Töpferware / Keramik' => DirectMarketerAttributes::CRAFTS_CERAMIC,
        'Kerzen' => DirectMarketerAttributes::CRAFTS_CANDLES,
        'Schmiedearbeiten' => DirectMarketerAttributes::CRAFTS_WROUGHT,
        'Schmuck' => DirectMarketerAttributes::CRAFTS_JEWELLERY,
        'Flechtarbeiten / Körbe' => DirectMarketerAttributes::CRAFTS_WEAVING,
        'Textilien & Textilkunst' => DirectMarketerAttributes::CRAFTS_TEXTILES,
        'Malerei & Skulpturen' => DirectMarketerAttributes::CRAFTS_ARTS,
        'Schnitzarbeiten' => DirectMarketerAttributes::CRAFTS_WOOD_CARVING,
        'Wolle / Wollpodukte' => DirectMarketerAttributes::CRAFTS_WOOL,
    ];

    public function getMainType(): int
    {
        return ArticleConstants::TYPE_DIRECT_MARKETER;
    }

    public function getExternalId(): string
    {
        return strip_tags((string) $this->object['remote_id']);
    }

    public function getName(): string
    {
        return strip_tags((string) $this->object['name']);
    }

    public function getDescription(): ?string
    {
        $description = null;
        $value = '';
        if (!\is_array($this->object['description'])) {
            return null;
        }
        if (array_key_exists('direkt_event', $this->object['description'])) {
            $value .= $this->object['description']['direkt_event']['text'];
        }
        if (array_key_exists('direkt', $this->object['description'])) {
            $value .= $this->object['description']['direkt']['text'];
        }


        if (!empty($value)) {
            $description = $value;
        }

        return $description ? $this->stringCleaner->purifyHtml((string) $description) : null;
    }

    public function getLatitude(): ?float
    {
        $latitude = $this->object['location']['Latitude'] ?? null;
        return $latitude ? (float) $latitude : null;
    }

    public function getLongitude(): ?float
    {
        $longitude = $this->object['location']['Longitude'] ?? null;
        return $longitude ? (float) $longitude : null;
    }

    /**
     * @return AddressAdapterInterface[]
     */
    public function getAddresses(): array
    {
        $mainAddress = $this->getMainAddress();
        return $mainAddress ? [ $mainAddress ] : [];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return new AddressAdapter($this->object);
    }

    public function getMainCategory(): ?CategoryAdapterInterface
    {
        $categories = $this->getCategories();

        return $categories[0] ?? null;
    }

    /**
     * @return CategoryAdapter[]
     */
    public function getCategories(): array
    {
        $category = new CategoryAdapter($this->object['main_category']);
        return [ $category ];
    }

    /**
     * @return MediumAdapter[]
     */
    public function getMedia(): array
    {
        return $this->cache('media', function () {
            return array_merge($this->extractImages(), $this->extractVideos());
        });
    }

    public function extractVideos(): array
    {
        $videos = [];

        foreach ($this->object['video'] ?? [] as $videoCollection) {
            if (is_array($videoCollection)) {
                foreach ($videoCollection as $videoToAdd) {
                    $video = null;
                    if ($videoToAdd && UrlUtility::isValidUrl($videoToAdd)) {
                        if (strpos($videoToAdd, 'youtu') !== false) {
                            $video = YoutubeVideoAdapter::create($videoToAdd);
                        }
                        if (strpos($videoToAdd, 'vimeo') !== false) {
                            $video = VimeoVideoAdapter::create($videoToAdd);
                        }
                        if ($video !== null) {
                            $videos[] = $video;
                        }
                    }
                }
            }
        }

        return $videos;
    }

    /** @return MediumAdapter[] */
    private function extractImages(): array
    {
        if (empty($this->object['images'] ?? [])) {
            return [];
        }

        if (array_key_exists('all', $this->object['images'])) {
            // Some images are nested inside 'all'.
            $items = $this->object['images']['all'];
        } else {
            // ... and some are not inside 'all'.
            $items = $this->object['images'];
        }

        $media = [];
        foreach ($items as $image) {
            $adapter = new MediumAdapter($image);
            if (UrlUtility::isValidUrl($adapter->getSourceUri())) {
                $media[] = $adapter;
            }
        }

        return $media;
    }

    public function hasAttributes(): bool
    {
        return !empty($this->parseAttributes());
    }

    public function getAttributes(): array
    {
        $attributes = AttributeImportUtility::splitAttributeArrayIntoImportables(
            $this->parseAttributes(),
            function ($name, $value) {
                return new AttributeAdapter($this->getExternalId(), $name, $value);
            }
        );

        return array_merge(
            $attributes,
            $this->extractAttributeWithValueMapping(
                'dv_special',
                GenericArticleAttributes::SERVICES,
                static::$SERVICES_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_marketing',
                DirectMarketerAttributes::SHOPPING,
                static::SHOPPING_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_decoration',
                DirectMarketerAttributes::DISTINCTIONS,
                static::DISTINCTION_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_fleisch',
                DirectMarketerAttributes::MEATS,
                static::MEAT_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_gefluegel_fisch',
                DirectMarketerAttributes::FISH,
                static::FISH_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_getraenke',
                DirectMarketerAttributes::DRINKS,
                static::DRINK_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_gemuese_obst',
                DirectMarketerAttributes::VEGETABLES_AND_FRUIT,
                static::VEGETABLE_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_aufstriche',
                DirectMarketerAttributes::OILS_SPREADS,
                static::OIL_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_milch',
                DirectMarketerAttributes::MILK,
                static::MILK_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_getreide',
                DirectMarketerAttributes::GRAINS,
                static::GRAIN_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_saison',
                DirectMarketerAttributes::SEASONALS,
                static::SEASONALS_MAPPING
            ),
            $this->extractAttributeWithValueMapping(
                'dv_prod_cat_handwerk',
                DirectMarketerAttributes::CRAFTS,
                static::CRAFTS_MAPPING
            )
        );
    }

    public function getFacebookUri(): ?string
    {
        return $this->getSocialMediaUri('facebook');
    }


    public function getTwitterUri(): ?string
    {
        return $this->getSocialMediaUri('twitter');
    }

    public function getInstagramUri(): ?string
    {
        return $this->getSocialMediaUri('instagram');
    }

    public function getYoutubeUri(): ?string
    {
        return $this->getSocialMediaUri('youtube');
    }

    public function getWikipediaUri(): ?string
    {
        return $this->getSocialMediaUri('wikipedia');
    }

    public function getFlickrUri(): ?string
    {
        return $this->getSocialMediaUri('flickr');
    }

    private function getSocialMediaUri(string $socialMedia): ?string
    {
        if (!array_key_exists('social_media', $this->object)
            || !\is_array($this->object['social_media'])
        ) {
            return null;
        }

        $data = $this->findInArray(
            $this->object['social_media'],
            'property',
            $socialMedia
        );

        return $data ? $data['value'] : null;
    }

    public function getSourceName(): ?string
    {
        $source = $this->object['headline_quellmandant'] ?? null;
        return $source ? strip_tags((string) $source) : null;
    }

    public function getAuthorName(): ?string
    {
        $author = $this->object['author'] ?? null;
        return $author ? strip_tags((string) $author) : null;
    }

    public function getBookingUris(): array
    {
        return [];
    }

    public function getDetailUri(): ?string
    {
        return null;
    }

    public function getOnlineStatus(): bool
    {
        return ($this->object['online'] ?? null) === '1';
    }

    public function getOpeningTimes(): ?string
    {
        return (string) json_encode($this->object['opentimes']);
    }

    public function getOpeningTimesFormat(): ?string
    {
        return ArticleConstants::OPENING_TIMES_FORMAT_LEGACY;
    }

    private function parseAttributes(): array
    {
        $attributes = $this->parseCommonToubizAttributes();
        $keys = [
            'dv_decoration' => DirectMarketerAttributes::AWARDS,
            'dv_decoration_regional' => DirectMarketerAttributes::REGIONAL_AWARDS,
        ];

        $valueOnlyKeys = [
            'panorama' => DirectMarketerAttributes::PANORAMA,
        ];

        foreach ($keys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key);
        }

        foreach ($valueOnlyKeys as $key => $name) {
            $attributes[$name] = $this->extractAttributes($key, true);
        }

        return $attributes;
    }


    private function extractAttributeWithValueMapping(
        string $toubizValueName,
        string $attributeName,
        array $mapping,
        bool $valueOnly = false
    ): array {
        $attributes = [];

        $values = (array) $this->extractAttributes($toubizValueName, $valueOnly);
        foreach ($values as $rawValue) {
            $value = $mapping[$rawValue] ?? null;
            if ($value === null) {
                $this->logger->warning(
                    sprintf(
                        'Cannot map %s/%s `%s` to a normalized value',
                        $attributeName,
                        $toubizValueName,
                        $rawValue
                    )
                );
                $value = $rawValue;
            }

            $attributes[] = new AttributeAdapter($this->getExternalId(), $attributeName, $value);
        }

        return $attributes;
    }


    private function extractAttributes(string $key, bool $valueOnly = false): ?array
    {
        $attributes = $this->object[$key]
            ?? $this->object['property'][$key]
            ?? $this->object['classification'][$key]
            ?? [];

        $normalizedAttributes = [];

        foreach ($attributes as $attribute) {
            if (!\is_array($attribute)) {
                continue;
            }
            if (array_key_exists('properties', $attribute)) {
                foreach ($attribute['properties'] as $property) {
                    $normalizedAttributes[] = $property;
                }
            } else {
                $normalizedAttributes[] = $attribute;
            }
        }

        if (empty($normalizedAttributes)) {
            return null;
        }

        return array_map(
            function (array $item) use ($valueOnly) {
                if ($valueOnly) {
                    return $item['value'];
                }
                return $item['text'];
            },
            $normalizedAttributes
        );
    }

    public function getAverageRating(): ?int
    {
        return null;
    }

    public function getNumberOfRatings(): ?int
    {
        return null;
    }

    public function getAbstract(): ?string
    {
        return null;
    }

    public function getFiles(): array
    {
        return [];
    }

    public function getCitySelectors(): array
    {
        $id = $this->object['id_location'] ?? null;

        if (\is_string($id)) {
            return [ new ExternalIdSelector(ExternalIdType::TOUBIZ_LEGACY, $id) ];
        }

        return [];
    }
}
