<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV2\ObjectAdapter;

use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Concern\ArticleConstants;
use Newland\Toubiz\Api\ObjectAdapter\AttributeAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\CityAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\DirectMarketerAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\GenericArticleAttributes;
use Newland\Toubiz\Api\ObjectAdapter\Attributes\PointOfInterestAttributes;
use Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter\ArticleAdapter as ArticleV1Adapter;
use Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter\AttributeAdapter;

class ArticleAdapter extends ArticleV1Adapter
{
    const REGULAR_ATTRIBUTE_MAPPING = [
        GenericArticleAttributes::CLIENT_ID => 'client.id',
        GenericArticleAttributes::CLIENT_NAME => 'client.name',
        GenericArticleAttributes::CLIENT_IMAGE => 'client.logo.url',
        GenericArticleAttributes::CLIENT_URL => 'client.url',
        GenericArticleAttributes::AUTHOR_NAME => 'legalInformation.author',

        GenericArticleAttributes::ADDITIONAL_SEARCH_STRING => 'seo.searchKeywords',
        CityAttributes::WEBCAM => 'contactInformation.webcam',
        DirectMarketerAttributes::PANORAMA => 'contactInformation.camera360',
        PointOfInterestAttributes::FREE_ENTRANCE => 'price.freeEntry',
        PointOfInterestAttributes::PRICE_INFORMATION => 'price.priceComment',
    ];

    const CATEGORY_SKI_AREA = '8a78311b-edcc-4a40-a350-96b86c202c67';

    public function getMainType(): int
    {
        // Ski areas used to be points but are now areas. We override this here to keep the legacy behavior
        // and see how it goes...
        if ($this->object['primaryCategory']['id'] === self::CATEGORY_SKI_AREA) {
            return ArticleConstants::TYPE_ATTRACTION;
        }

        return $this->type;
    }

    public function getName(): string
    {
        return $this->object['text']['name'];
    }

    public function getAbstract(): ?string
    {
        return $this->object['text']['abstract'];
    }

    public function getDescription(): ?string
    {
        return $this->object['text']['description'];
    }

    public function getLatitude(): ?float
    {
        return $this->object['latitude'];
    }

    public function getLongitude(): ?float
    {
        return $this->object['longitude'];
    }

    public function getMainAddress(): ?AddressAdapterInterface
    {
        return null;
    }

    public function getAddresses(): array
    {
        return [];
    }


    protected function getFixedAttributes(): array
    {
        $attributes = [
            $this->getPriceAttribute(),
        ];

        return array_filter($attributes);
    }

    protected function getPriceAttribute(): ?AttributeAdapterInterface
    {
        $currencies = $this->object['price']['currencies'] ?? [];
        $priceGroups = $this->object['price']['priceGroups'] ?? [];
        if (empty($priceGroups)) {
            return null;
        }

        $table = [];
        foreach ($priceGroups as $priceGroup) {
            $table[] = [ sprintf('<strong>%s</strong>', $priceGroup['title']) ];
            foreach ($priceGroup['priceEntries'] ?? [] as $priceEntry) {
                $row = [ $priceEntry['title'] ];
                foreach ($currencies as $currency) {
                    if ($currency === 'eur') {
                        $row[] = sprintf('%.2f €', $priceEntry['eur']);
                    } elseif ($currency === 'chf') {
                        $row[] = sprintf('CHF %.2f', $priceEntry['chf']);
                    } elseif ($priceEntry[$currency] ?? null) {
                        $row[] = sprintf('%s %.2f', $currency, $priceEntry[$currency]);
                    }
                }
                $row[] = $priceEntry['comment'];
                $table[] = $row;
            }
        }

        return new AttributeAdapter(
            PointOfInterestAttributes::PRICES,
            $table,
            null,
            $this->getExternalId() . '__prices'
        );
    }


    public function getZipCodes(): array
    {
        return $this->object['zipCodes'] ?? [];
    }

    public function getClaim(): string
    {
        return $this->object['areaText']['claim'] ?? '';
    }

    public function getFacts(): string
    {
        return $this->object['areaText']['facts'] ?? '';
    }

    public function getSourceApiVersion(): ?string
    {
        return '2';
    }
}
