<?php declare(strict_types=1);

namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter;

use DateTime;
use DateTimeInterface;
use DateTimeZone;
use Newland\Toubiz\Api\Constants\EventScope;
use Newland\Toubiz\Api\ObjectAdapter\AbstractObjectAdapter;
use Newland\Toubiz\Api\ObjectAdapter\AddressAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\CategoryAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventDateAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\EventTagAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasLocationDataInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasMarketingPriceInterface;
use Newland\Toubiz\Api\ObjectAdapter\HasSeoInformationInterface;
use Newland\Toubiz\Api\ObjectAdapter\MediumAdapterInterface;
use Newland\Toubiz\Api\ObjectAdapter\SeoInformationAdapterInterface;
use Newland\Toubiz\Api\Service\LanguageAware;

class EventAdapter extends AbstractObjectAdapter implements
    EventAdapterInterface,
    HasLocationDataInterface,
    HasMarketingPriceInterface,
    HasSeoInformationInterface
{
    use LanguageAware,
        EventCommon,
        HasMarketingPrice,
        WithCitySelectors;

    protected const SCOPE_MAPPING = [
        'local' => EventScope::SCOPE_LOCAL,
        'regional' => EventScope::SCOPE_REGIONAL,
        'multiregional' => EventScope::SCOPE_TRANSREGIONAL,
    ];

    /** @var string[] */
    private $detailUriTemplates = [];

    public function setDetailUriTemplates(array $detailUriTemplates): void
    {
        $this->detailUriTemplates = $detailUriTemplates;
    }

    public function getDataSource(): string
    {
        return self::DATA_SOURCE_TOUBIZ_APIV1;
    }

    public function getEventType(): string
    {
        return $this->object['eventType'];
    }

    public function getName(): string
    {
        return $this->object['name'];
    }

    public function getDescription(): string
    {
        return $this->object['description'];
    }

    public function getAdditionalInformation(): string
    {
        return $this->object['intro'];
    }

    public function getBeginsAt(): ?DateTimeInterface
    {
        if ($this->object['firstDate']) {
            return $this->buildDateTimeFromStrings(
                $this->object['firstDate']['date'],
                $this->object['firstDate']['startAt']
            );
        }

        return null;
    }

    public function getEndsAt(): ?DateTimeInterface
    {
        if ($this->object['lastDate']) {
            return $this->buildDateTimeFromStrings(
                $this->object['lastDate']['date'],
                $this->object['lastDate']['endAt']
            );
        }

        return null;
    }

    public function getAdmission(): string
    {
        return $this->object['additionalBookingInformation'];
    }

    public function getIsHighlight(): bool
    {
        return $this->object['highlight'];
    }

    public function getIsTipp(): bool
    {
        return $this->object['highlight'];
    }

    public function getTicketUri(): ?string
    {
        return $this->object['bookingUrl'] ?: null;
    }

    public function getTicketContact(): ?string
    {
        return $this->object['bookingHotline'] ?: null;
    }

    public function getLink(): ?string
    {
        return ($this->object['contactInformation']['website'] ?? null) ?: null;
    }

    public function getUpdatedAt(): DateTimeInterface
    {
        return new DateTime();
    }

    /** @return CategoryAdapterInterface[] */
    public function getCategories(): array
    {
        return [ new CategoryAdapter($this->object['category']) ];
    }

    /** @return EventDateAdapterInterface[] */
    public function getEventDates(): array
    {
        $adapters = [];

        $rceIdEvent = $this->object['externalIds']['rceId'] ?? null;
        $detailUriTemplate = $this->detailUriTemplates['rce'] ?? null;

        foreach ($this->object['dates'] ?? [] as $date) {
            $date['_meta']['rceIdEvent'] = $rceIdEvent;
            $eventDateAdapter = new EventDateAdapter($date);
            $rceIdEventDate = $date['externalIds']['rceId'] ?? null;

            if ($rceIdEvent && $rceIdEventDate && $detailUriTemplate) {
                $eventDateAdapter->setDetailUriTemplate($detailUriTemplate);
            }

            $adapters[] = $eventDateAdapter;
        }

        return $adapters;
    }

    /** @return EventTagAdapterInterface[] */
    public function getEventTags(): array
    {
        return array_map(
            function ($tag) {
                return new EventTagAdapter($tag);
            },
            $this->object['tags']
        );
    }

    public function getAttributes(): array
    {
        // TODO: Implement getAttributes() method.
        return [];
    }

    /** @return MediumAdapterInterface[] */
    public function getMedia(): array
    {
        return array_map(
            function ($media) {
                return new MediumAdapter($media);
            },
            $this->object['media']
        );
    }

    public function getMainMedium(): ?MediumAdapterInterface
    {
        $medium = $this->object['mainImage']
            ?? $this->object['media'][0]
            ?? null;

        if (!$medium) {
            return null;
        }

        return new MediumAdapter($medium);
    }

    public function getOrganizer(): ?AddressAdapterInterface
    {
        return null;
    }

    public function getLocation(): ?AddressAdapterInterface
    {
        $location = ($this->object['location'] ?? $this->object['locationData']) ?? null;
        if (!$location || !$location['address']) {
            return null;
        }

        return new ArticleBasedAddressAdapter($location, $location['address']);
    }

    public function getSourceName(): ?string
    {
        return $this->object['author'];
    }

    public function getScope(): int
    {
        return static::SCOPE_MAPPING[$this->object['type']] ?? EventScope::SCOPE_UNDEFINED;
    }

    public function getAdditionalSearchString(): ?string
    {
        return $this->object['seo']['searchKeywords'] ?? null;
    }

    public function getExternalId(): string
    {
        return $this->object['id'];
    }

    public function getIsCanceled(): bool
    {
        return $this->object['canceled'];
    }

    public function getAreaName(): ?string
    {
        if ($this->object['location'] ?? null) {
            if ($this->object['location']['type'] === 'point') {
                return $this->object['location']['address']['city'] ?? null;
            }
            if ($this->object['location']['type'] === 'area') {
                return $this->object['location']['name'];
            }
            return null;
        }

        return $this->object['locationData']['address']['city'] ?? null;
    }

    public function getCanonicalUrl(): ?string
    {
        return $this->object['seo']['canonicalLink'] ?? null;
    }

    public function getSeoInformation(): ?SeoInformationAdapterInterface
    {
        if ($this->object['seo'] ?? null) {
            return new SeoInformationAdapter($this->object['seo']);
        }

        return null;
    }

    public function getLastModifiedAt(): ?DateTime
    {
        $updatedAt = $this->object['updatedAt'] ?? $this->object['createdAt'];
        $dateTime = DateTime::createFromFormat('Y-m-d\TH:i:s', $updatedAt);
        if (!$dateTime) {
            return null;
        }

        return $dateTime->setTimezone(new DateTimeZone('Europe/Berlin'));
    }

    public function getPriceInformation(): ?array
    {
        return $this->object['price'];
    }
}
