<?php declare(strict_types=1);


namespace Newland\Toubiz\Api\Service\Toubiz\ApiV1;

use GuzzleHttp\Promise\EachPromise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Uri;
use Newland\Toubiz\Api\Exception\ItemNotFoundException;
use Newland\Toubiz\Api\Service\AbstractService;
use Newland\Toubiz\Api\Service\LanguageAware;
use Newland\Toubiz\Api\Service\ServiceResult;
use Newland\Toubiz\Api\Service\Toubiz\ApiV1\ObjectAdapter\EventAdapter;

class EventService extends AbstractService
{
    use LanguageAware;

    const DEFAULT_BASE_URI = 'https://mein.toubiz.de';

    /** @var string[] */
    private $detailUriTemplates = [];

    public function setDetailUriTemplates(array $detailUriTemplates): void
    {
        $this->detailUriTemplates = $detailUriTemplates;
    }

    public function fetchEvents(callable $block): ServiceResult
    {
        $ids = $this->listAllIds();
        $total = count($ids);

        $pool = new EachPromise($this->requestDetailPages($ids), [
            'concurrency' => $this->parameters['concurrency'] ?? 10,
            'fulfilled' => function ($data) use ($block, $total) {
                $eventAdapter = $this->bodyToAdapter($data);
                $block($eventAdapter, $total);
            },
        ]);

        $pool->promise()->wait();

        $result = new ServiceResult();
        $result->setAll($ids);
        return $result;
    }

    public function fetchEvent(string $id, callable $importCallable, callable $deleteCallable): ServiceResult
    {
        $query = $this->getDetailQueryParams();

        $this->requestDetailPage($query, $id)
            ->then(
                function ($data) use ($importCallable) {
                    $eventAdapter = $this->bodyToAdapter($data);
                    $importCallable($eventAdapter);
                },
                function ($data) use ($deleteCallable, $id) {
                    if ($data instanceof ItemNotFoundException) {
                        $deleteCallable($id);
                    }
                }
            )
            ->wait();

        $result = new ServiceResult();
        $result->setAll([ $id ]);
        return $result;
    }

    private function bodyToAdapter(?array $body): ?EventAdapter
    {
        if ($body === null) {
            return null;
        }

        $eventAdapter = new EventAdapter($body);
        $eventAdapter->setDetailUriTemplates($this->detailUriTemplates);

        return $eventAdapter;
    }

    private function listAllIds(): array
    {
        $query = array_replace_recursive($this->parameters['request'] ?? [], [
            'api_token' => $this->apiKey,
            'minimal' => true,
            'language' => $this->language,
        ]);

        if ($this->delta !== null && $this->delta->asYears() < 1) {
            $query['filter'] = $query['filter'] ?? [];
            $query['filter']['updatedAfter'] = (new \DateTime())->sub($this->delta)->format('Y-m-d\TH:i:s');
        }

        $uri = (new Uri('api/v1/event'))->withQuery(http_build_query($query));

        return $this->jsonRequest($uri, 5, function (array $data) {
            return empty($data['_errors']);
        })->then(function (array $data) {
            return array_keys($data['payload']);
        })->wait();
    }

    private function requestDetailPages(array $ids): \Generator
    {
        $query = $this->getDetailQueryParams();

        foreach ($ids as $id) {
            yield $this->requestDetailPage($query, $id);
        }
    }

    private function requestDetailPage(array $query, string $id): PromiseInterface
    {
        $uri = (new Uri(sprintf('api/v1/event/%s', $id)))->withQuery(http_build_query($query));
        return $this->jsonRequest($uri, 5, function ($data) {
            return empty($data['_errors']);
        })->then(function ($data) {
            return $data['payload'];
        });
    }

    public function getDetailQueryParams(): array
    {
        return [
            'language' => $this->language,
            'api_token' => $this->apiKey,
            'include' => [
                'awards',
                'channelContext',
                'client',
                'contactInformation',
                'dates',
                'externalIds',
                'fieldBlueprints',
                'fieldValues',
                'files',
                'location',
                'mainImage',
                'media',
                'partners',
                'price',
                'printInformation',
                'relatedArticles',
                'seo',
                'tags',
            ],
            'sizes' => [
                'preview' => $this->parameters['previewSize'] ?? '600',
            ],
        ];
    }
}
